//	TorusGamesRenderer.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesRenderer.h"
#import "TorusGames-Common.h"
#import "TorusGamesGPUDefinitions.h"
#import "GeometryGamesUtilities-Common.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesColorSpaces.h"
#import "GeometryGamesLocalization.h"	//	for IsCurrentLanguage()
#import "GeometryGamesSound.h"			//	for gPlaySounds
#import "GeometryGamesMatrix44.h"
#import <MetalKit/MetalKit.h>


#define CROSSWORD_CHARACTER_TEXTURE_SIZE	256
#define WORDSEARCH_CHARACTER_TEXTURE_SIZE	256

//		Refinement levels for ball, tube and slice meshes
#define MESH_REFINEMENT_LEVEL_FOR_HIGH_RESOLUTION	4	//	high resolution for ViewBasicLarge
#define MESH_REFINEMENT_LEVEL_FOR_LOW_RESOLUTION	3	//	low  resolution for ViewRepeating


typedef enum
{
	Texture2DBackground,
	Texture2DKleinBottleAxis,
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	Texture2DHandFlat,
	Texture2DHandGrab,
#endif

	Texture2DIntroSpriteBaseIndex,
	Texture2DIntroSprite0 = Texture2DIntroSpriteBaseIndex,
	Texture2DIntroSprite1,
	Texture2DIntroSprite2,
	Texture2DIntroSprite3,
	
	Texture2DTicTacToeGrid,
	Texture2DTicTacToeMarkerX,
	Texture2DTicTacToeMarkerO,
	Texture2DTicTacToeWinLine,
	
	Texture2DGomokuGrid,
	Texture2DGomokuStoneBlack,
	Texture2DGomokuStoneWhite,
	Texture2DGomokuWinLine,
	
	Texture2DMazeMaze,
	Texture2DMazeMouse,
	Texture2DMazeCheese,
	
	Texture2DCrosswordFirstCell,
	Texture2DCrosswordLastCell = Texture2DCrosswordFirstCell + (MAX_CROSSWORD_SIZE*MAX_CROSSWORD_SIZE - 1),
	Texture2DCrosswordPendingCharacter,	//	used for Korean jamo input, even on iOS
	Texture2DCrosswordArrow,

	Texture2DWordSearchFirstCell,
	Texture2DWordSearchLastCell = Texture2DWordSearchFirstCell + (MAX_WORDSEARCH_SIZE*MAX_WORDSEARCH_SIZE - 1),
	Texture2DWordSearchMark,
	Texture2DWordSearchHotSpot,
	
	Texture2DJigsawCollageWithoutBorders,
	Texture2DJigsawCollageWithBorders,
	
	Texture2DChessHalo,
	Texture2DChessArrow,
	Texture2DChessTarget,
	Texture2DChessCheckmate,
	Texture2DChessStalemate,
	Texture2DChessWhitePieceBaseIndex,
	Texture2DChessWhiteKing	= Texture2DChessWhitePieceBaseIndex,
	Texture2DChessWhiteQueen,
	Texture2DChessWhiteBishop,
	Texture2DChessWhiteKnight,
	Texture2DChessWhiteRook,
	Texture2DChessWhitePawn,
	Texture2DChessBlackPieceBaseIndex,
	Texture2DChessBlackKing	= Texture2DChessBlackPieceBaseIndex,
	Texture2DChessBlackQueen,
	Texture2DChessBlackBishop,
	Texture2DChessBlackKnight,
	Texture2DChessBlackRook,
	Texture2DChessBlackPawn,
	Texture2DChessDialBase,
	Texture2DChessDialSweep,
	Texture2DChessDialRim,
	Texture2DChessDialArrow,
	
	Texture2DPoolBallBaseIndex,
	Texture2DPoolBall0 = Texture2DPoolBallBaseIndex,
	Texture2DPoolBall1,
	Texture2DPoolBall2,
	Texture2DPoolBall3,
	Texture2DPoolBall4,
	Texture2DPoolBall5,
	Texture2DPoolBall6,
	Texture2DPoolBall7,
	Texture2DPoolPocket,
	Texture2DPoolStick,
	Texture2DPoolLineOfSight,
	Texture2DPoolHotSpot,
	
	Texture2DApplesApple,
	Texture2DApplesWorm,
	Texture2DApplesFaceHappy,
	Texture2DApplesFaceYucky,
	Texture2DApplesNumeralBaseIndex,
	Texture2DApplesNumeral0 = Texture2DApplesNumeralBaseIndex,	//	Texture2DApplesNumeral0 is unused, but keeps the indexing simple
	Texture2DApplesNumeral1,
	Texture2DApplesNumeral2,
	Texture2DApplesNumeral3,
	Texture2DApplesNumeral4,
	Texture2DApplesNumeral5,
	Texture2DApplesNumeral6,
	Texture2DApplesNumeral7,
	Texture2DApplesNumeral8,
	
	TotalNumTextures
} TextureIndex;


typedef struct
{
	TorusGames2DWorldData	itsWorldData;
	float					itsBackgroundTexReps;		//	integer value, but stored as float for convenience of GPU
	simd_float3x3			itsKleinAxisPlacementA,
							itsKleinAxisPlacementB;
	faux_simd_half4			itsKleinAxisColorA,					//	premultiplied alpha
							itsKleinAxisColorB;					//	premultiplied alpha
	float					itsKleinAxisTexReps;		//	integer value, but stored as float for convenience of GPU
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	simd_float3x3			itsHandCursorPlacement;
#endif

	faux_simd_half4			itsGomokuWinLineColor,				//	premultiplied alpha
							itsMazeWallColor;					//	premultiplied alpha
	float					itsCrosswordGridTexReps,	//	integer value, but stored as float for convenience of GPU
							itsCrosswordGridLineTransitionZoneWidthInverse;	//	sorry about the long name!
	faux_simd_half4			itsCrosswordFlashColor,				//	premultiplied alpha
							itsCrosswordGridColor,				//	premultiplied alpha
							itsCrosswordHotCellColor,			//	premultiplied alpha
							itsCrosswordHotWordColor,			//	premultiplied alpha
							itsCrosswordCharacterColor,			//	premultiplied alpha
							itsCrosswordPendingCharacterColor,	//	premultiplied alpha
							itsCrosswordArrowColor,				//	premultiplied alpha
							itsWordSearchCharacterColor,		//	premultiplied alpha
							itsWordSearchMarkedWordColors[MAX_WORDSEARCH_NUM_WORDS],	//	premultiplied alpha
							itsWordSearchPendingWordColor,		//	premultiplied alpha
							itsWordSearchHotPointColor;			//	premultiplied alpha
	simd_float4				itsJigsawTexturePlacement[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE];	//	( u_min, v_min, u_max - u_min, v_max - v_min ) in bottom-up coordinates
	faux_simd_half4			itsChessWhoseTurnHaloColor,			//	premultiplied alpha
							itsChessGhostColor,					//	premultiplied alpha (partially transparent white)
							itsChessDragPieceHaloColor,			//	premultiplied alpha
							itsChessDialBaseColor,				//	premultiplied alpha
							itsChessDialSweepColor,				//	premultiplied alpha
							itsChessDialRimColor,				//	premultiplied alpha
							itsChessDialArrowColor,				//	premultiplied alpha
							itsPoolHotSpotColor,				//	premultiplied alpha
							itsApplesNumeralColor;				//	premultiplied alpha
} TorusGames2DUniformData;

typedef struct
{
	TorusGames3DWorldData	itsWorldData;

	float					itsPatternParityPlus,	//	= +1.0; for quarter-turn and half-turn patterns
							itsPatternParityMinus;	//	= -1.0; for quarter-turn and half-turn patterns
	simd_float2				itsTexCoordShifts[6];	//	array of six pairs (∆s, ∆t) to let the texture move
													//		with the game cell rather than staying fixed
													//		on the frame cell wall
	faux_simd_half4			itsWallColors[NUM_AVAILABLE_WALL_COLORS];

	faux_simd_half4			itsTicTacToeNeutralColor,	//	premultiplied alpha
							itsTicTacToeXColor,			//	premultiplied alpha
							itsTicTacToeOColor,			//	premultiplied alpha
							itsTicTacToeWinLineColor;	//	premultiplied alpha

	faux_simd_half4			itsMazeTracksColor,			//	premultiplied alpha
							itsMazeSliderColor,			//	premultiplied alpha
							itsMazeGoalOuterColor,		//	premultiplied alpha
							itsMazeGoalInnerColor;		//	premultiplied alpha

} TorusGames3DUniformData;


//	ARC can't handle object references within C structs,
//	so package up the following data in Objective-C objects instead.

@interface MeshBufferPair : NSObject
{
@public
	unsigned int	itsNumFaces;
	id<MTLBuffer>	itsVertexBuffer,
					itsIndexBuffer;
}
@end
@implementation MeshBufferPair
@end

@interface TransformationBufferPair : NSObject
{
@public
	id<MTLBuffer>	itsPlainTransformations,
					itsReflectingTransformations;
}
@end
@implementation TransformationBufferPair
@end


static simd_float3x3	ConvertPlacementToSIMD(const Placement2D *aPlacement);


//	Privately-declared methods
@interface TorusGamesRenderer()

- (void)   setUpPipelineStates;
- (void)shutDownPipelineStates;
- (void)   setUpDepthStencilState;
- (void)shutDownDepthStencilState;
- (void)   setUpFixedBuffers;
- (void)shutDownFixedBuffers;
- (void)   setUpInflightBuffers;
- (void)shutDownInflightBuffers;
- (void)   setUpTexturesWithModelData:(ModelData *)md;
- (void)shutDownTextures;
- (void)   setUpSamplers;
- (void)shutDownSamplers;

- (id<MTLTexture>)makeTicTacToeGridTextureWithColor:(ColorP3Linear)aColor
					size:(NSUInteger)aSize borderWidth:(NSUInteger)aBorderWidth;
- (id<MTLTexture>)makeGomokuGridTextureWithColor:(ColorP3Linear)aColor
					size:(NSUInteger)aSize lineHalfWidth:(NSUInteger)aLineHalfWidth;
- (NSString *)jigsawPuzzleSourceImageNameWithModelData:(ModelData *)md;
- (void)makeJigsawPieceCollagesWithSourceImage:(id<MTLTexture>)aSourceImage
					pieceTemplate:(id<MTLTexture>)aPieceTemplate modelData:(ModelData *)md;
- (void)computeJigsawCollageLayout:(JigsawCollageLayout *)aCollageLayout
					sourceImageSize:(NSUInteger)aSourceSizePx
					pieceTemplateWidth:(NSUInteger)aPieceTemplateWidthPx
					pieceTemplateHeight:(NSUInteger)aPieceTemplateHeightPx
					modelData:(ModelData *)md;

- (MeshBufferPair *)makeCube;
- (MeshBufferPair *)makeCubeSkeletonOuterFaces;
- (MeshBufferPair *)makeCubeSkeletonInnerFaces;
- (MeshBufferPair *)makeBallWithRefinementLevel:(unsigned int)aRefinementLevel;
- (MeshBufferPair *)makeTubeWithRefinementLevel:(unsigned int)aRefinementLevel;
- (id<MTLBuffer>)makeCircularSliceWithRefinementLevel:(unsigned int)aRefinementLevel;

- (id<MTLBuffer>)write2DUniformsIntoBuffer:(id<MTLBuffer>)aUniformsBuffer modelData:(ModelData *)md;
- (id<MTLBuffer>)write2DCoveringTransformationsIntoBuffer:(id<MTLBuffer>)aCoveringTransformationBuffer modelData:(ModelData *)md;
- (id<MTLBuffer>)write2DSpritePlacementsIntoBuffer:(id<MTLBuffer>)aSpritePlacementBuffer modelData:(ModelData *)md;

- (id<MTLBuffer>)write3DUniformsIntoBuffer:(id<MTLBuffer>)aUniformsBuffer modelData:(ModelData *)md;
- (TransformationBufferPair *)write3DCoveringTransformationsIntoBuffersPlain:(id<MTLBuffer>)aPlainCoveringTransformationBuffer
									reflecting:(id<MTLBuffer>)aReflectingCoveringTransformationBuffer modelData:(ModelData *)md;
- (id<MTLBuffer>)write3DPolyhedronPlacementsIntoBuffer:(id<MTLBuffer>)aPolyhedronPlacementBuffer modelData:(ModelData *)md;

- (id<MTLTexture>)makeMazeMaskWithModelData:(ModelData *)md;
- (id<MTLTexture>)makeCrosswordCharacterMask:(Char16)aCharacter;
- (id<MTLTexture>)makeWordSearchCharacterMask:(Char16)aCharacter;
- (id<MTLTexture>)makeApplesNumeralMask:(Char16)aCharacter;

- (void)			  encode2DCommandsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							  inflightDataBuffers:	(NSDictionary<NSString *, id> *)someInflightDataBuffers
										modelData:	(ModelData *)md;

- (void) encodeCommandsFor2DBackgroundWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations;

- (void)  encodeCommandsFor2DKleinAxesWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations;

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
- (void) encodeCommandsFor2DHandCursorWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
										modelData:	(ModelData *)md;
#endif

- (void)	  encodeCommandsFor2DIntroWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
							spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)  encodeCommandsFor2DTicTacToeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)     encodeCommandsFor2DGomokuWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)       encodeCommandsFor2DMazeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)  encodeCommandsFor2DCrosswordWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void) encodeCommandsFor2DWordSearchWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)	 encodeCommandsFor2DJigsawWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)	  encodeCommandsFor2DChessWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)	   encodeCommandsFor2DPoolWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;

- (void)	 encodeCommandsFor2DApplesWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								    uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						    spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
										modelData:	(ModelData *)md;


- (void)			  encode3DCommandsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							  inflightDataBuffers:	(NSDictionary<NSString *, id> *)someInflightDataBuffers
										modelData:	(ModelData *)md;

- (void)	  encodeCommandsFor3DWallsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
										modelData:	(ModelData *)md;

- (void)  encodeCommandsFor3DTicTacToeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
										modelData:	(ModelData *)md;

- (void)	   encodeCommandsFor3DMazeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
									uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
					   numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
										modelData:	(ModelData *)md;

@end


@implementation TorusGamesRenderer
{
	id<MTLBuffer>					//	for 2D games
									its2DUniformBuffers[NUM_INFLIGHT_BUFFERS],
									its2DCoveringTransformationBuffers[NUM_INFLIGHT_BUFFERS],
									its2DSpritePlacementBuffers[NUM_INFLIGHT_BUFFERS],
	
									//	for 3D games
									its3DUniformBuffers[NUM_INFLIGHT_BUFFERS],
									its3DPlainCoveringTransformationBuffers[NUM_INFLIGHT_BUFFERS],
									its3DReflectingCoveringTransformationBuffers[NUM_INFLIGHT_BUFFERS],
									its3DPolyhedronPlacementBuffers[NUM_INFLIGHT_BUFFERS];
	
	id<MTLRenderPipelineState>		itsRenderPipelineState2DBackground,
									itsRenderPipelineState2DSpriteWithColor,
									itsRenderPipelineState2DSpriteWithTexture,
									itsRenderPipelineState2DSpriteWithTextureSubset,
									itsRenderPipelineState2DSpriteWithColorAndRGBATexture,
									itsRenderPipelineState2DSpriteWithColorAndMask,
									itsRenderPipelineState2DLineWithRGBATexture,
									itsRenderPipelineState2DLineWithEndcapsRGBATexture,
									itsRenderPipelineState2DLineWithEndcapsAndMask,
									itsRenderPipelineState2DGrid,
									itsRenderPipelineState3DWallPlain,
									itsRenderPipelineState3DWallReflection,
									itsRenderPipelineState3DWallQuarterTurn,
									itsRenderPipelineState3DWallHalfTurn,
									itsRenderPipelineState3DPolyhedron,
									itsRenderPipelineState3DPolyhedronClippedToFrameCell,
									itsRenderPipelineState3DPolyhedronSlice,
									itsRenderPipelineState3DPolyhedronSliceClippedToFrameCell,
									itsRenderPipelineState3DPolyhedronSliceClippedToWinLine,
									itsRenderPipelineState3DPolyhedronSliceClippedToFrameCellAndWinLine;
	id<MTLComputePipelineState>		itsMakeTicTacToeGridPipelineState,
									itsMakeGomokuGridPipelineState,
									itsMakeMazeMaskPipelineState,
									itsMakeJigsawCollagesPipelineState;

	id<MTLDepthStencilState>		its3DDepthStencilStateNormal,
									its3DDepthStencilStateAlwaysPasses;

	id<MTLBuffer>					its2DSquareVertexBuffer,
									its2DLineWithEndcapsVertexBuffer,
									its3DWallVertexBuffer,
									its3DWallWithApertureVertexBuffer,
									its3DWallPlacements;
	MeshBufferPair					*its3DCubeMesh,
									*its3DCubeSkeletonOuterFaceMesh,
									*its3DCubeSkeletonInnerFaceMesh,
									*its3DBallMeshHighRes,	//	for ViewBasicLarge
									*its3DBallMeshLowRes,	//	for ViewRepeating
									*its3DTubeMeshHighRes,	//	for ViewBasicLarge
									*its3DTubeMeshLowRes;	//	for ViewRepeating
	id<MTLBuffer>					its3DSquareSliceVertexBuffer,
									its3DCircularSliceVertexBufferHighRes,	//	for ViewBasicLarge
									its3DCircularSliceVertexBufferLowRes;	//	for ViewRepeating

	id<MTLTexture>					itsTextures[TotalNumTextures];

	id<MTLSamplerState>				itsTextureSamplerIsotropicRepeat,
									itsTextureSamplerIsotropicClampToEdge;
	
	//	Remember what character each Crossword texture represents,
	//	so when the user taps a key we'll know which texture needs updating.
	Char16							itsCrosswordTextureCharacters[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE];
}


- (id)initWithLayer:(CAMetalLayer *)aLayer device:(id<MTLDevice>)aDevice
	multisampling:(bool)aMultisamplingFlag depthBuffer:(bool)aDepthBufferFlag stencilBuffer:(bool)aStencilBufferFlag
{
	self = [super initWithLayer:aLayer device:aDevice
		multisampling:aMultisamplingFlag depthBuffer:aDepthBufferFlag stencilBuffer:aStencilBufferFlag
		mayExportWithTransparentBackground:false];
	if (self != nil)
	{
	}
	return self;
}


#pragma mark -
#pragma mark graphics setup/shutdown

- (void)setUpGraphicsWithModelData:(ModelData *)md
{
	//	This method gets called once when the view is first created, and then
	//	again whenever the app comes out of background and re-enters the foreground.
	//
	//		Technical note:  An added advantage of setting up the graphics here,
	//		instead of in initWithLayer:…, is that we avoid calling [self …]
	//		from within the initializer.  In the present implementation,
	//		calls to [self …] from the initializer would be harmless,
	//		but the risk is that if we ever wrote a subclass of this class,
	//		then such calls would be received by a not-yet-fully-initialized object.
	//
	
	[super setUpGraphicsWithModelData:md];

	[self setUpPipelineStates];
	[self setUpDepthStencilState];
	[self setUpFixedBuffers];
	[self setUpInflightBuffers];
	[self setUpTexturesWithModelData:md];
	[self setUpSamplers];

	//	There's no urgent need to redraw the game immediately,
	//	but it seems like a good thing to do anyhow,
	//	so if there are ever any problems, they'll be visible immediately.
	//
	md->itsChangeCount++;
}

- (void)shutDownGraphicsWithModelData:(ModelData *)md
{
	//	This method gets called when the view
	//	leaves the foreground and goes into the background.
	//	There's no need to call it when the view gets deallocated,
	//	because at that point ARC automatically releases all
	//	the view's Metal objects.

	[self shutDownPipelineStates];
	[self shutDownDepthStencilState];
	[self shutDownFixedBuffers];
	[self shutDownInflightBuffers];
	[self shutDownTextures];
	[self shutDownSamplers];
	
	[super shutDownGraphicsWithModelData:md];
}

- (void)setUpPipelineStates
{
	id<MTLLibrary>				theGPUFunctionLibrary;
	MTLFunctionConstantValues	*theCompileTimeConstants;
	NSError						*theError;
	id<MTLFunction>				theGPUVertexFunction2DBackground,
								theGPUVertexFunction2DSpritePlain,
								theGPUVertexFunction2DSpriteWithTexture,
								theGPUVertexFunction2DLine,
								theGPUVertexFunction2DLineWithEndcaps,
								theGPUVertexFunction2DSpriteWithTextureSubset,
								theGPUVertexFunction3DWall,
								theGPUVertexFunction3DPolyhedron,
								theGPUVertexFunction3DPolyhedronClippedToFrameCell,
								theGPUVertexFunction3DPolyhedronSlice,
								theGPUVertexFunction3DPolyhedronSliceClippedToFrameCell,
								theGPUVertexFunction3DPolyhedronSliceClippedToWinLine,
								theGPUVertexFunction3DPolyhedronSliceClippedToFrameCellAndWinLine,
								theGPUFragmentFunction2DColor,
								theGPUFragmentFunction2DTexture,
								theGPUFragmentFunction2DColorWithRGBATexture,
								theGPUFragmentFunction2DColorWithMask,
								theGPUFragmentFunction2DGrid,
								theGPUFragmentFunction3DWallPlain,
								theGPUFragmentFunction3DWallReflection,
								theGPUFragmentFunction3DWallQuarterTurn,
								theGPUFragmentFunction3DWallHalfTurn,
								theGPUFragmentFunction3DPolyhedron,
								theGPUComputeFunctionMakeTicTacToeGrid,
								theGPUComputeFunctionMakeGomokuGrid,
								theGPUComputeFunctionMakeMazeMask,
								theGPUComputeFunctionMakeJigsawCollages;
	NSString					*theCollageFunctionName,
								*theMazeMaskFunctionName;
	MTLVertexDescriptor			*the2DVertexDescriptor,
								*the2DOffsettableVertexDescriptor,
								*the3DWallVertexDescriptor,
								*the3DPolyhedronVertexDescriptor;
	unsigned int				theMultisamplingNumSamples;
	MTLPixelFormat				theDepthAttachmentPixelFormat,
								theStencilAttachmentPixelFormat;
	MTLRenderPipelineDescriptor	*thePipelineDescriptor;
	
	const uint8_t	theFalseValue	= false,
					theTrueValue	= true;


	//	Apple's documentation for MTLFunctionConstantValues says that
	//
	//		A single MTLFunctionConstantValues object can be applied
	//		to multiple MTLFunction objects (for example, a vertex function
	//		and a fragment function). After a specialized function has been created,
	//		any changes to its constant values have no further effect on it.
	//		However, you can reset, add, or modify any constant values
	//		in the MTLFunctionConstantValues object and reuse it
	//		to create another MTLFunction object.
	//

	theGPUFunctionLibrary = [itsDevice newDefaultLibrary];
	
	theGPUVertexFunction2DBackground					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DBackground"				];
	theGPUVertexFunction2DSpritePlain					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DSpritePlain"				];
	theGPUVertexFunction2DSpriteWithTexture				= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DSpriteWithTexture"			];
	theGPUVertexFunction2DLine							= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DLine"						];
	theGPUVertexFunction2DLineWithEndcaps				= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DLineWithEndcaps"			];
	theGPUVertexFunction2DSpriteWithTextureSubset		= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction2DSpriteWithTextureSubset"	];

	theGPUVertexFunction3DWall							= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DWall"						];

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedron						= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedronClippedToFrameCell		= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedronSlice					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedronSliceClippedToFrameCell	= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedronSliceClippedToWinLine	= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gClipToFrameCell"		];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gDrawSlice"			];
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gClipSliceToWinLine"	];
	theGPUVertexFunction3DPolyhedronSliceClippedToFrameCellAndWinLine	= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesVertexFunction3DPolyhedron"
																constantValues:theCompileTimeConstants error:&theError];
#ifdef DEBUG
	GEOMETRY_GAMES_ASSERT(theError == nil, "GPU function compilation error or warning");
#endif

	theGPUFragmentFunction2DColor					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction2DColor"				];
	theGPUFragmentFunction2DTexture					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction2DTexture"				];
	theGPUFragmentFunction2DColorWithRGBATexture	= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction2DColorWithRGBATexture"];
	theGPUFragmentFunction2DColorWithMask			= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction2DColorWithMask"		];
	theGPUFragmentFunction2DGrid					= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction2DGrid"				];

	theGPUFragmentFunction3DWallPlain				= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction3DWallPlain"			];
	theGPUFragmentFunction3DWallReflection			= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction3DWallReflection"		];
	theGPUFragmentFunction3DWallQuarterTurn			= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction3DWallQuarterTurn"		];
	theGPUFragmentFunction3DWallHalfTurn			= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction3DWallHalfTurn"		];

	theGPUFragmentFunction3DPolyhedron				= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesFragmentFunction3DPolyhedron"			];


	//	Create a MTLVertexDescriptor describing the vertex attributes
	//	that the GPU sends along to each vertex.  Each vertex attribute
	//	may be a 1-, 2-, 3- or 4-component vector.

	
	//	Ordinary 2D vertices
	
	the2DVertexDescriptor = [MTLVertexDescriptor vertexDescriptor];

	[[the2DVertexDescriptor attributes][VertexAttribute2DPosition] setFormat:MTLVertexFormatFloat2];
	[[the2DVertexDescriptor attributes][VertexAttribute2DPosition] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the2DVertexDescriptor attributes][VertexAttribute2DPosition] setOffset:offsetof(TorusGames2DVertexData, pos)];

	[[the2DVertexDescriptor attributes][VertexAttribute2DTexCoords] setFormat:MTLVertexFormatFloat2];
	[[the2DVertexDescriptor attributes][VertexAttribute2DTexCoords] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the2DVertexDescriptor attributes][VertexAttribute2DTexCoords] setOffset:offsetof(TorusGames2DVertexData, tex)];

	[[the2DVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStepFunction:MTLVertexStepFunctionPerVertex];
	[[the2DVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStride:sizeof(TorusGames2DVertexData)];


	//	Offsettable 2D vertices
	
	the2DOffsettableVertexDescriptor = [MTLVertexDescriptor vertexDescriptor];
	
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DPosition] setFormat:MTLVertexFormatFloat2];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DPosition] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DPosition] setOffset:offsetof(TorusGames2DOffsetableVertexData, pos)];

	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DTexCoords] setFormat:MTLVertexFormatFloat2];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DTexCoords] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DTexCoords] setOffset:offsetof(TorusGames2DOffsetableVertexData, tex)];

	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DOffset] setFormat:MTLVertexFormatFloat];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DOffset] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the2DOffsettableVertexDescriptor attributes][VertexAttribute2DOffset] setOffset:offsetof(TorusGames2DOffsetableVertexData, off)];

	[[the2DOffsettableVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStepFunction:MTLVertexStepFunctionPerVertex];
	[[the2DOffsettableVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStride:sizeof(TorusGames2DOffsetableVertexData)];


	//	3D wall vertices
	
	the3DWallVertexDescriptor = [MTLVertexDescriptor vertexDescriptor];
	
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallPosition] setFormat:MTLVertexFormatFloat3];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallPosition] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallPosition] setOffset:offsetof(TorusGames3DWallVertexData, pos)];

	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallTexCoords] setFormat:MTLVertexFormatFloat2];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallTexCoords] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallTexCoords] setOffset:offsetof(TorusGames3DWallVertexData, tex)];

	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallWeight] setFormat:MTLVertexFormatFloat2];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallWeight] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the3DWallVertexDescriptor attributes][VertexAttribute3DWallWeight] setOffset:offsetof(TorusGames3DWallVertexData, wgt)];

	[[the3DWallVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStepFunction:MTLVertexStepFunctionPerVertex];
	[[the3DWallVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStride:sizeof(TorusGames3DWallVertexData)];
	
	
	//	3D polyhedron vertices
	
	the3DPolyhedronVertexDescriptor = [MTLVertexDescriptor vertexDescriptor];
	
	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronPosition] setFormat:MTLVertexFormatFloat3];
	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronPosition] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronPosition] setOffset:offsetof(TorusGames3DPolyhedronVertexData, pos)];

	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronNormal] setFormat:MTLVertexFormatHalf3];
	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronNormal] setBufferIndex:BufferIndexVFVertexAttributes];
	[[the3DPolyhedronVertexDescriptor attributes][VertexAttribute3DPolyhedronNormal] setOffset:offsetof(TorusGames3DPolyhedronVertexData, nor)];

	[[the3DPolyhedronVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStepFunction:MTLVertexStepFunctionPerVertex];
	[[the3DPolyhedronVertexDescriptor layouts][BufferIndexVFVertexAttributes] setStride:sizeof(TorusGames3DPolyhedronVertexData)];
	

	//	Pipeline state constants
	theMultisamplingNumSamples		= (itsMultisamplingFlag ? METAL_MULTISAMPLING_NUM_SAMPLES	: 1						);
	theDepthAttachmentPixelFormat	= (itsDepthBufferFlag   ? MTLPixelFormatDepth32Float		: MTLPixelFormatInvalid	);
	theStencilAttachmentPixelFormat	= (itsStencilBufferFlag ? MTLPixelFormatStencil8			: MTLPixelFormatInvalid	);


	//	Create MTLRenderPipelineStates.

	//	Note:
	//	Alpha blending determines how the final fragment
	//	blends in with the previous color buffer contents.
	//	For opaque surfaces we can disable blending.
	//	For partially transparent surfaces we may enable blending
	//	but must take care to draw the scene in back-to-front order.
	//	The comment accompanying the definition of PREMULTIPLY_RGBA()
	//	explains the (1, 1 - α) blend factors.
	//
	//	Caution:  For the 3D games, avoid alpha-blending on iOS,
	//	because the PowerVR GPU's tile-based deferred rendering
	//	works more efficiently with opaque surfaces.
	//	If alpha-blending is ever needed in the 3D games,
	//	draw all opaque content first, then draw transparent content.
	//	(As well as drawing opaque and transparent content,
	//	a third technique -- alpha-test with possible "discard" --
	//	is also possible, but this is even less efficient than alpha-blending.
	//	If alpha-test/discard is unavoidable, then Imagination Technologies
	//	recommends drawing in the order (1) opaque, (2) alpha-test/discard
	//	and (3) transparent, which makes sense.)
	
	//	Less-important note:
	//	It's not clear whether -newRenderPipelineStateWithDescriptor:
	//	retains thePipelineDescriptor as an object, or whether it merely
	//	takes a snapshot of the data it contains and then forgets the object itself.
	//	In the latter case we could simplify the following code by modifying
	//	and re-using the same pipeline descriptor.
	//	But obviously that would be a disaster if the first pipeline state object
	//	were keeping a reference to the pipeline descriptor as an object.
	
	//	2D background
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D background"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DBackground	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DTexture	];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState2DBackground = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D sprite with color
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D sprite with color"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DSpritePlain];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColor	];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DSpriteWithColor = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D sprite with texture
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D sprite with texture"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DSpriteWithTexture	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DTexture			];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DSpriteWithTexture = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D sprite with texture subset
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D jigsaw piece"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DSpriteWithTextureSubset];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DTexture				];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DSpriteWithTextureSubset = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D sprite with partially transparent texture
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D sprite with partially transparent texture"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DSpriteWithTexture		];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColorWithRGBATexture	];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DSpriteWithColorAndRGBATexture = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D sprite with color and alpha texture
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D sprite with color and alpha texture"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DSpriteWithTexture	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColorWithMask	];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DSpriteWithColorAndMask = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D line
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D line"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DLine					];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColorWithRGBATexture	];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DLineWithRGBATexture = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D line with endcaps and colored RGBA texture
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D line with endcaps and colored RGBA texture"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DLineWithEndcaps		];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColorWithRGBATexture	];
	[thePipelineDescriptor setVertexDescriptor:the2DOffsettableVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DLineWithEndcapsRGBATexture = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D line with endcaps and colored alpha-only texture
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D line with endcaps and colored alpha-only texture"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DLineWithEndcaps];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DColorWithMask];
	[thePipelineDescriptor setVertexDescriptor:the2DOffsettableVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DLineWithEndcapsAndMask = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	2D grid
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 2D grid"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction2DBackground	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction2DGrid		];
	[thePipelineDescriptor setVertexDescriptor:the2DVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];

	//	See comments on alpha blending above
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:YES];
	[[thePipelineDescriptor colorAttachments][0] setRgbBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setAlphaBlendOperation:MTLBlendOperationAdd];
	[[thePipelineDescriptor colorAttachments][0] setSourceRGBBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setSourceAlphaBlendFactor:MTLBlendFactorOne];
	[[thePipelineDescriptor colorAttachments][0] setDestinationRGBBlendFactor:MTLBlendFactorOneMinusSourceAlpha];
	[[thePipelineDescriptor colorAttachments][0] setDestinationAlphaBlendFactor:MTLBlendFactorOneMinusSourceAlpha];

	itsRenderPipelineState2DGrid = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	3D wall with plain coloring
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D wall with plain coloring"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DWall		];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DWallPlain];
	[thePipelineDescriptor setVertexDescriptor:the3DWallVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DWallPlain = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	3D wall with reflection coloring
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D wall with reflection coloring"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DWall				];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DWallReflection	];
	[thePipelineDescriptor setVertexDescriptor:the3DWallVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DWallReflection = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	3D wall with quarter-turn coloring
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D wall with quarter-turn coloring"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DWall				];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DWallQuarterTurn	];
	[thePipelineDescriptor setVertexDescriptor:the3DWallVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DWallQuarterTurn = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//	3D wall with half-turn coloring
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D wall with half-turn coloring"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DWall			];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DWallHalfTurn	];
	[thePipelineDescriptor setVertexDescriptor:the3DWallVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DWallHalfTurn = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//	3D game content
	
	//		the 3D content itself

	//			For ViewRepeating, no custom clipping is needed.
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedron		];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron	];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedron = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//			For ViewBasicLarge, clip all polyhedra to the frame cell, whose corners sit at (±1/2, ±1/2, ±1/2).
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron with clipping"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedronClippedToFrameCell	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron					];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedronClippedToFrameCell = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];
	
	//		the slices where the 3D content intersects the frame cell walls

	//			for rectangular and circular cross sections
	
	//				for ViewRepeating, not clipped to frame cell
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron slice"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedronSlice];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron	];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedronSlice = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//				for ViewBasicLarge, clip all polyhedra to the frame cell, whose corners sit at (±1/2, ±1/2, ±1/2).
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron slice"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedronSliceClippedToFrameCell	];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron						];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedronSliceClippedToFrameCell = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//			for elliptical cross sections, which get clipped to the win-line tube's ends
	
	//				for ViewRepeating, not clipped to frame cell
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron slice"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedronSliceClippedToWinLine];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron					];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedronSliceClippedToWinLine = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//				for ViewBasicLarge, clip all polyhedra to the frame cell, whose corners sit at (±1/2, ±1/2, ±1/2).
	
	thePipelineDescriptor = [[MTLRenderPipelineDescriptor alloc] init];
	[thePipelineDescriptor setLabel:@"Torus Games render pipeline for 3D polyhedron slice"];
	[thePipelineDescriptor setSampleCount:theMultisamplingNumSamples];
	[thePipelineDescriptor setVertexFunction:  theGPUVertexFunction3DPolyhedronSliceClippedToFrameCellAndWinLine];
	[thePipelineDescriptor setFragmentFunction:theGPUFragmentFunction3DPolyhedron								];
	[thePipelineDescriptor setVertexDescriptor:the3DPolyhedronVertexDescriptor];
	[thePipelineDescriptor setDepthAttachmentPixelFormat:theDepthAttachmentPixelFormat];
	[thePipelineDescriptor setStencilAttachmentPixelFormat:theStencilAttachmentPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setPixelFormat:itsColorPixelFormat];
	[[thePipelineDescriptor colorAttachments][0] setBlendingEnabled:NO];

	itsRenderPipelineState3DPolyhedronSliceClippedToFrameCellAndWinLine = [itsDevice newRenderPipelineStateWithDescriptor:thePipelineDescriptor error:NULL];

	//	Game-specific compute pipeline states
	//
	//		Note:  If desired we could load the compute pipeline state
	//		only for the currently active game, and provide a mechanism
	//		(analogous to -refreshTexturesWithModelData:) to re-load
	//		the compute pipeline state when the user changes to a new game.
	//		But I'm guessing that these pipeline states are much less
	//		memory intensive than the games' texture sets, and that
	//		there's no harm in sticking with the simpler approach
	//		of keeping all three compute pipeline states available
	//		at all times.
	//

	theGPUComputeFunctionMakeTicTacToeGrid	= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesComputeFunctionMakeTicTacToeGrid"];

	theGPUComputeFunctionMakeGomokuGrid		= [theGPUFunctionLibrary newFunctionWithName:@"TorusGamesComputeFunctionMakeGomokuGrid"   ];

#if TARGET_OS_IOS
	if (@available(iOS 13.0, *))
	{
		//	We're running on an A8 GPU or higher,
		//	so our GPU function may safely use 16-bit integer arithmetic.
		theMazeMaskFunctionName = @"TorusGamesComputeFunctionMakeMazeMask";
	}
	else
	{
		//	We might be running on an A7 GPU (or lower?),
		//	so fall back to a legacy GPU function
		//	that uses 32-bit integer arithmetic.
		theMazeMaskFunctionName = @"TorusGamesComputeFunctionMakeMazeMaskLEGACY";
	}
#else	//	macOS
	//	Use the GPU function with 16-bit integer arithmetic, and hope for the best.
	theMazeMaskFunctionName = @"TorusGamesComputeFunctionMakeMazeMask";
#endif
	theGPUComputeFunctionMakeMazeMask		= [theGPUFunctionLibrary newFunctionWithName:theMazeMaskFunctionName];

#if TARGET_OS_IOS
	if (@available(iOS 13.0, *))
	{
		//	We're running on an A8 GPU or higher,
		//	so our GPU function may safely use 16-bit integer arithmetic.
		theCollageFunctionName = @"TorusGamesComputeFunctionMakeJigsawCollages";
	}
	else
	{
		//	We might be running on an A7 GPU (or lower?),
		//	so fall back to a legacy GPU function
		//	that uses 32-bit integer arithmetic.
		theCollageFunctionName = @"TorusGamesComputeFunctionMakeJigsawCollagesLEGACY";
	}
#else	//	macOS
	//	Use the GPU function with 16-bit integer arithmetic, and hope for the best.
	theCollageFunctionName = @"TorusGamesComputeFunctionMakeJigsawCollages";
#endif
	theCompileTimeConstants = [[MTLFunctionConstantValues alloc] init];
#ifdef MAKE_GAME_CHOICE_ICONS
	[theCompileTimeConstants setConstantValue:&theTrueValue  type:MTLDataTypeBool withName:@"gMakeGameChoiceIcons"];
#else
	[theCompileTimeConstants setConstantValue:&theFalseValue type:MTLDataTypeBool withName:@"gMakeGameChoiceIcons"];
#endif
	theGPUComputeFunctionMakeJigsawCollages = [theGPUFunctionLibrary newFunctionWithName:theCollageFunctionName
												constantValues:theCompileTimeConstants error:&theError];

	itsMakeTicTacToeGridPipelineState	= [itsDevice newComputePipelineStateWithFunction:theGPUComputeFunctionMakeTicTacToeGrid  error:NULL];
	itsMakeGomokuGridPipelineState		= [itsDevice newComputePipelineStateWithFunction:theGPUComputeFunctionMakeGomokuGrid     error:NULL];
	itsMakeMazeMaskPipelineState		= [itsDevice newComputePipelineStateWithFunction:theGPUComputeFunctionMakeMazeMask	     error:NULL];
	itsMakeJigsawCollagesPipelineState	= [itsDevice newComputePipelineStateWithFunction:theGPUComputeFunctionMakeJigsawCollages error:NULL];
}

- (void)shutDownPipelineStates
{
	itsRenderPipelineState2DBackground									= nil;
	itsRenderPipelineState2DSpriteWithColor								= nil;
	itsRenderPipelineState2DSpriteWithTexture							= nil;
	itsRenderPipelineState2DSpriteWithTextureSubset						= nil;
	itsRenderPipelineState2DSpriteWithColorAndRGBATexture				= nil;
	itsRenderPipelineState2DSpriteWithColorAndMask						= nil;
	itsRenderPipelineState2DLineWithRGBATexture							= nil;
	itsRenderPipelineState2DLineWithEndcapsRGBATexture					= nil;
	itsRenderPipelineState2DLineWithEndcapsAndMask						= nil;
	itsRenderPipelineState2DGrid										= nil;
	
	itsRenderPipelineState3DWallPlain									= nil;
	itsRenderPipelineState3DWallReflection								= nil;
	itsRenderPipelineState3DWallQuarterTurn								= nil;
	itsRenderPipelineState3DWallHalfTurn								= nil;
	
	itsRenderPipelineState3DPolyhedron									= nil;
	itsRenderPipelineState3DPolyhedronClippedToFrameCell				= nil;
	itsRenderPipelineState3DPolyhedronSlice								= nil;
	itsRenderPipelineState3DPolyhedronSliceClippedToFrameCell			= nil;
	itsRenderPipelineState3DPolyhedronSliceClippedToWinLine				= nil;
	itsRenderPipelineState3DPolyhedronSliceClippedToFrameCellAndWinLine	= nil;
	
	itsMakeTicTacToeGridPipelineState									= nil;
	itsMakeGomokuGridPipelineState										= nil;
	itsMakeMazeMaskPipelineState										= nil;
	itsMakeJigsawCollagesPipelineState									= nil;
}

- (void)setUpDepthStencilState
{
	MTLDepthStencilDescriptor	*theDepthStencilDescriptor;

	theDepthStencilDescriptor = [[MTLDepthStencilDescriptor alloc] init];
	[theDepthStencilDescriptor setDepthCompareFunction:MTLCompareFunctionLess];
	[theDepthStencilDescriptor setDepthWriteEnabled:YES];
	its3DDepthStencilStateNormal		= [itsDevice newDepthStencilStateWithDescriptor:theDepthStencilDescriptor];

	theDepthStencilDescriptor = [[MTLDepthStencilDescriptor alloc] init];
	[theDepthStencilDescriptor setDepthCompareFunction:MTLCompareFunctionAlways];
	[theDepthStencilDescriptor setDepthWriteEnabled:YES];
	its3DDepthStencilStateAlwaysPasses	= [itsDevice newDepthStencilStateWithDescriptor:theDepthStencilDescriptor];
}

- (void)shutDownDepthStencilState
{
	its3DDepthStencilStateNormal		= nil;
	its3DDepthStencilStateAlwaysPasses	= nil;
}

- (void)setUpFixedBuffers
{
	unsigned int	i;
	simd_float4x4	theWallPlacements[NUM_FRAME_WALLS];

	//	All the fixed buffers together use only a small amount of memory,
	//	so to keep the program logic simple, leave all of them (2D and 3D both)
	//	set up at all times.
	
	
	//	2D

	//	square
	its2DSquareVertexBuffer	= [itsDevice
		newBufferWithBytes:	gSquareVertices
		length:				sizeof(gSquareVertices)
		options:			MTLResourceStorageModeShared];

	//	line with endcaps
	its2DLineWithEndcapsVertexBuffer = [itsDevice
		newBufferWithBytes:	gLineWithEndcapsOffsettableVertices
		length:				sizeof(gLineWithEndcapsOffsettableVertices)
		options:			MTLResourceStorageModeShared];
	
	
	//	3D
	//
	//	Note:  Metal's default value for frontFacingWinding is MTLWindingClockwise.
	
	//	wall
	its3DWallVertexBuffer = [itsDevice
		newBufferWithBytes:	gWallVertices
		length:				sizeof(gWallVertices)
		options:			MTLResourceStorageModeShared];
	
	//	wall with aperture
	its3DWallWithApertureVertexBuffer = [itsDevice
		newBufferWithBytes:	gWallWithApertureVertices
		length:				sizeof(gWallWithApertureVertices)
		options:			MTLResourceStorageModeShared];
	
	//	wall-into-framecell placements
	for (i = 0; i < NUM_FRAME_WALLS; i++)
		theWallPlacements[i] = ConvertMatrix44ToSIMD(gWallIntoFrameCellPlacements[i]);
	its3DWallPlacements = [itsDevice
		newBufferWithBytes:	theWallPlacements
		length:				sizeof(theWallPlacements)
		options:			MTLResourceStorageModeShared];
	
	//	cube for 3D Tic-Tac-Toe
	its3DCubeMesh = [self makeCube];
	
	//	cube skeleton for 3D Maze goal
	its3DCubeSkeletonOuterFaceMesh = [self makeCubeSkeletonOuterFaces];
	its3DCubeSkeletonInnerFaceMesh = [self makeCubeSkeletonInnerFaces];
	
	//	ball for 3D Tic-Tac-Toe and 3D Maze
	its3DBallMeshHighRes	= [self makeBallWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_HIGH_RESOLUTION];
	its3DBallMeshLowRes		= [self makeBallWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_LOW_RESOLUTION ];

	//	win line for 3D Tic-Tac-Toe
	its3DTubeMeshHighRes	= [self makeTubeWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_HIGH_RESOLUTION];
	its3DTubeMeshLowRes		= [self makeTubeWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_LOW_RESOLUTION ];
	
	//	square cross-sectional slice
	its3DSquareSliceVertexBuffer = [itsDevice
		newBufferWithBytes:	gSquareSliceVertices
		length:				sizeof(gSquareSliceVertices)
		options:			MTLResourceStorageModeShared];
	
	//	circular cross-sectional slice
	its3DCircularSliceVertexBufferHighRes	= [self makeCircularSliceWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_HIGH_RESOLUTION];
	its3DCircularSliceVertexBufferLowRes	= [self makeCircularSliceWithRefinementLevel:MESH_REFINEMENT_LEVEL_FOR_LOW_RESOLUTION ];
}

- (void)shutDownFixedBuffers
{
	its2DSquareVertexBuffer					= nil;
	its2DLineWithEndcapsVertexBuffer		= nil;
	
	its3DWallVertexBuffer					= nil;
	its3DWallWithApertureVertexBuffer		= nil;
	its3DWallPlacements						= nil;
	
	its3DCubeMesh							= nil;
	its3DCubeSkeletonOuterFaceMesh			= nil;
	its3DCubeSkeletonInnerFaceMesh			= nil;
	its3DBallMeshHighRes					= nil;
	its3DBallMeshLowRes						= nil;
	its3DTubeMeshHighRes					= nil;
	its3DTubeMeshLowRes						= nil;
	
	its3DSquareSliceVertexBuffer			= nil;
	its3DCircularSliceVertexBufferHighRes	= nil;
	its3DCircularSliceVertexBufferLowRes	= nil;
	
}

- (void)setUpInflightBuffers
{
	unsigned int	i;
	
	for (i = 0; i < NUM_INFLIGHT_BUFFERS; i++)
	{
		its2DUniformBuffers[i]							= nil;	//	will be created in -write2DUniformsIntoBuffer:modelData:
		its2DCoveringTransformationBuffers[i]			= nil;	//	will be created in -write2DCoveringTransformationsIntoBuffer:modelData:
		its2DSpritePlacementBuffers[i]					= nil;	//	will be created in -write2DSpritePlacementsIntoBuffer:modelData:

		its3DUniformBuffers[i]							= nil;	//	will be created in -write3DUniformsIntoBuffer:modelData:
		its3DPlainCoveringTransformationBuffers[i]		= nil;	//	will be created in -write3DCoveringTransformationsIntoBuffersPlain:reflecting:modelData:
		its3DReflectingCoveringTransformationBuffers[i]	= nil;	//	will be created in -write3DCoveringTransformationsIntoBuffersPlain:reflecting:modelData:
		its3DPolyhedronPlacementBuffers[i]				= nil;	//	will be created in -write3DPolyhedronPlacementsIntoBuffer:modelData:
	}
}

- (void)shutDownInflightBuffers
{
	unsigned int	i;
	
	for (i = 0; i < NUM_INFLIGHT_BUFFERS; i++)
	{
		its2DUniformBuffers[i]							= nil;
		its2DCoveringTransformationBuffers[i]			= nil;
		its2DSpritePlacementBuffers[i]					= nil;

		its3DUniformBuffers[i]							= nil;
		its3DPlainCoveringTransformationBuffers[i]		= nil;
		its3DReflectingCoveringTransformationBuffers[i]	= nil;
		its3DPolyhedronPlacementBuffers[i]				= nil;
	}
}

- (void)setUpTexturesWithModelData:(ModelData *)md
{
	MTKTextureLoader							*theTextureLoader;
	NSDictionary<MTKTextureLoaderOption, id>	*theTextureLoaderOptions,
												*theTextureLoaderOptionsWithWrite;
	bool										theDeviceSupportsTextureRoughing;
	NSError										*theError = nil;
	
	static const double	theRoughingFactor =
#ifdef MAKE_GAME_CHOICE_ICONS
							0.00;	//	use smooth background for tiny menu icons
#else
							0.25;	//	use slightly roughened background in the game itself
#endif
	
	theTextureLoader		= [[MTKTextureLoader alloc] initWithDevice:itsDevice];
	theTextureLoaderOptions	=
	@{
		MTKTextureLoaderOptionTextureUsage			:	@(MTLTextureUsageShaderRead),
		MTKTextureLoaderOptionTextureStorageMode	:	@(MTLStorageModePrivate),
		MTKTextureLoaderOptionAllocateMipmaps		:	@(YES),
		MTKTextureLoaderOptionGenerateMipmaps		:	@(YES)	//	-newTextureWithName:… ignores this option
																//		(as confirmed in its documentation).
																//	Instead, the Asset Catalog provides the mipmaps.
	};
	theTextureLoaderOptionsWithWrite	=
	@{
		//	Textures that we'll want to "roughen" with a GPU compute function
		//	will need MTLTextureUsageShaderWrite as well as MTLTextureUsageShaderRead.
		MTKTextureLoaderOptionTextureUsage			:	@(MTLTextureUsageShaderRead | MTLTextureUsageShaderWrite),
		MTKTextureLoaderOptionTextureStorageMode	:	@(MTLStorageModePrivate),
		MTKTextureLoaderOptionAllocateMipmaps		:	@(YES),
		MTKTextureLoaderOptionGenerateMipmaps		:	@(YES)	//	-newTextureWithName:… ignores this option
																//		(as confirmed in its documentation).
																//	Instead, the Asset Catalog provides the mipmaps.
	};

	//	GeometryGamesComputeFunctionRoughenTexture()
	//	requires a read_write texture.
	//
	//		Note:  On macOS, Apple Silicon Macs provide
	//		the required read_write textures, but Intel Macs
	//		don't let GPU compute functions write to mipmap levels
	//		at all (well, except for the base level).
	//
	theDeviceSupportsTextureRoughing = ( [itsDevice readWriteTextureSupport] >= MTLReadWriteTextureTier2 );


	itsTextures[Texture2DKleinBottleAxis]	= [theTextureLoader
												newTextureWithName:@"common/LineDashed"
												scaleFactor:1.0
												bundle:nil
												options:theTextureLoaderOptions
												error:&theError];

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	itsTextures[Texture2DHandFlat]			= [theTextureLoader
												newTextureWithName:@"common/HandFlat"
												scaleFactor:1.0
												bundle:nil
												options:theTextureLoaderOptions
												error:&theError];

	itsTextures[Texture2DHandGrab]			= [theTextureLoader
												newTextureWithName:@"common/HandGrab"
												scaleFactor:1.0
												bundle:nil
												options:theTextureLoaderOptions
												error:&theError];
#endif

	switch (md->itsGame)
	{
		case GameNone:
			break;

		case Game2DIntro:

			itsTextures[Texture2DBackground]	= [theTextureLoader
													newTextureWithName:@"Intro/Sand"
													scaleFactor:1.0
													bundle:nil
													options:theTextureLoaderOptions
													error:&theError];

			itsTextures[Texture2DIntroSprite0]	= [theTextureLoader
													newTextureWithName:@"Intro/Flounder"
													scaleFactor:1.0
													bundle:nil
													options:theTextureLoaderOptions
													error:&theError];

			itsTextures[Texture2DIntroSprite1]	= [theTextureLoader
													newTextureWithName:@"Intro/Blue Crab"
													scaleFactor:1.0
													bundle:nil
													options:theTextureLoaderOptions
													error:&theError];

			itsTextures[Texture2DIntroSprite2]	= [theTextureLoader
													newTextureWithName:@"Intro/Pink Scallop"
													scaleFactor:1.0
													bundle:nil
													options:theTextureLoaderOptions
													error:&theError];

			itsTextures[Texture2DIntroSprite3]	= [theTextureLoader
													newTextureWithName:@"Intro/Japanese Scallop"
													scaleFactor:1.0
													bundle:nil
													options:theTextureLoaderOptions
													error:&theError];

			break;
		
		case Game2DTicTacToe:

#if (SHAPE_OF_SPACE_TICTACTOE == 1)
			//	Use a lighter background, with no roughing.
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:(ColorP3Linear){0.97, 0.97, 0.97, 1.00} size:512];
#elif (SHAPE_OF_SPACE_TICTACTOE == 2)
			//	Use no background texture at all -- we want X's and O's over a transparent background.
			itsTextures[Texture2DBackground]		= nil;
#elif defined(MAKE_GAME_CHOICE_ICONS)
			//	For better contrast with the Choose-A-Game panel's
			//	pure white background, darken the Tic-Tac-Toe background.
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:(ColorP3Linear){0.93, 0.89, 0.86, 1.00} size:512];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];
#elif defined(MAKE_APP_ICON)
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:(ColorP3Linear){1.00, 0.96, 0.92, 1.00} size:512];
			//	no roughing for app icon
#else
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:(ColorP3Linear){1.00, 0.96, 0.92, 1.00} size:512];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];
#endif

			itsTextures[Texture2DTicTacToeGrid]		= [self makeTicTacToeGridTextureWithColor:(ColorP3Linear){0.125, 0.125, 0.125, 1.000}
														size:512 borderWidth:16];
#if (defined(MAKE_GAME_CHOICE_ICONS) || defined(MAKE_APP_ICON))
//	When MAKE_GAME_CHOICE_ICONS or MAKE_APP_ICON is enabled,
//	unroughened grid needs to be darker (also in other games?)
			itsTextures[Texture2DTicTacToeGrid]		= [self makeTicTacToeGridTextureWithColor:(ColorP3Linear){0.0625, 0.0625, 0.0625, 1.000}
														size:512 borderWidth:16];
#else
			[self roughenTexture:itsTextures[Texture2DTicTacToeGrid] roughingFactor:1.0];
#endif

			itsTextures[Texture2DTicTacToeMarkerX]	= [theTextureLoader
														newTextureWithName:
#ifdef MAKE_APP_ICON
															@"Tic-Tac-Toe/MarkerXDark"
#else
															(theDeviceSupportsTextureRoughing ?
																@"Tic-Tac-Toe/MarkerX" :
																@"Tic-Tac-Toe/MarkerXDark")
#endif
														scaleFactor:1.0
														bundle:nil
														options:(theDeviceSupportsTextureRoughing ?
																	theTextureLoaderOptionsWithWrite :
																	theTextureLoaderOptions)
														error:&theError];
			[self roughenTexture:itsTextures[Texture2DTicTacToeMarkerX] roughingFactor:1.0];

			itsTextures[Texture2DTicTacToeMarkerO]	= [theTextureLoader
														newTextureWithName:@"Tic-Tac-Toe/MarkerO"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptionsWithWrite
														error:&theError];
			[self roughenTexture:itsTextures[Texture2DTicTacToeMarkerO] roughingFactor:1.0];

			itsTextures[Texture2DTicTacToeWinLine]	= [self makeRGBATextureOfColor:(ColorP3Linear){0.0, 0.875, 0.0, 1.0}
														width:1024 height:64];
			[self roughenTexture:itsTextures[Texture2DTicTacToeWinLine] roughingFactor:0.125];
			break;
		
		case Game2DGomoku:

			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:(ColorP3Linear){0.98, 0.78, 0.53, 1.00} size:256];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];

			itsTextures[Texture2DGomokuGrid]		= [self makeGomokuGridTextureWithColor:(ColorP3Linear){0.125, 0.125, 0.125, 1.000}
														size:256 lineHalfWidth:8];
#ifdef MAKE_GAME_CHOICE_ICONS
#else
			[self roughenTexture:itsTextures[Texture2DGomokuGrid] roughingFactor:1.0];
#endif

			itsTextures[Texture2DGomokuStoneBlack]	= [theTextureLoader
														newTextureWithName:@"Gomoku/StoneBlack"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DGomokuStoneWhite]	= [theTextureLoader
														newTextureWithName:@"Gomoku/StoneWhite"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DGomokuWinLine]		= [theTextureLoader
														newTextureWithName:@"Gomoku/WinLine"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			break;
		
		case Game2DMaze:

#ifdef MAKE_GAME_CHOICE_ICONS
			//	For better contrast with the Choose-A-Game panel's
			//	pure white background, let the Maze's background be light grey.
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:
#warning clean this up
													//		(ColorP3Linear){1.0000, 0.9375, 0.8750, 1.0000}
													//		(ColorP3Linear){0.8750, 0.8750, 0.8750, 1.0000}
															(ColorP3Linear){0.9375, 0.9375, 0.9375, 1.0000}
														size:256];
#else
			//	A roughened white background looks good in the Maze itself.
			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:
															(ColorP3Linear){1.0000, 1.0000, 1.0000, 1.0000}
														size:256];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];
#endif

			itsTextures[Texture2DMazeMaze]			= [self makeMazeMaskWithModelData:md];

			itsTextures[Texture2DMazeMouse]			= [theTextureLoader
														newTextureWithName:@"Maze/Mouse"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DMazeCheese]		= [theTextureLoader
														newTextureWithName:@"Maze/Cheese"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			break;
		
		case Game2DCrossword:
			{
				unsigned int	thePuzzleSize,
								h,
								v,
								theTextureIndex;
				Char16			(*theBoard)[MAX_CROSSWORD_SIZE],
								(*theSolution)[MAX_CROSSWORD_SIZE],
								thePendingCharacter;

				itsTextures[Texture2DBackground] = [self makeRGBATextureOfColor:(ColorP3Linear){1.0, 1.0, 1.0, 1.0} size:1];

				thePuzzleSize		= md->itsGameOf.Crossword2D.itsPuzzleSize;
				theBoard			= md->itsGameOf.Crossword2D.itsBoard;
				theSolution			= md->itsGameOf.Crossword2D.itsSolution;
				thePendingCharacter	= md->itsGameOf.Crossword2D.itsPendingCharacter;

				//	Cells
				for (h = 0; h < thePuzzleSize; h++)
				{
					for (v = 0; v < thePuzzleSize; v++)
					{
						theTextureIndex = Texture2DCrosswordFirstCell + h*thePuzzleSize + v;

						if (theSolution[h][v] != L'*'	//	not a black cell
						 &&    theBoard[h][v] != L' ')	//	not an empty cell
						{
							itsTextures[theTextureIndex]		= [self makeCrosswordCharacterMask:theBoard[h][v]];
							itsCrosswordTextureCharacters[h][v]	= theBoard[h][v];
						}
						else
						{
							itsTextures[theTextureIndex]		= nil;
							itsCrosswordTextureCharacters[h][v]	= 0;
						}
					}
				}

				//	Pending character (if any)
				if (thePendingCharacter != NO_PENDING_CHARACTER)
					itsTextures[Texture2DCrosswordPendingCharacter] = [self makeCrosswordCharacterMask:thePendingCharacter];
				else
					itsTextures[Texture2DCrosswordPendingCharacter] = nil;

				//	Word-direction arrow
				itsTextures[Texture2DCrosswordArrow] = [theTextureLoader
														newTextureWithName:@"Crossword/DirectionArrow"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];
			}
			break;
		
		case Game2DWordSearch:
			{
				unsigned int	thePuzzleSize,
								h,
								v;
				Char16			(*theBoard)[MAX_WORDSEARCH_SIZE];

				itsTextures[Texture2DBackground] = [self makeRGBATextureOfColor:(ColorP3Linear){1.0, 1.0, 1.0, 1.0} size:256];
				[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];

				thePuzzleSize	= md->itsGameOf.WordSearch2D.itsPuzzleSize;
				theBoard		= md->itsGameOf.WordSearch2D.itsBoard;

				//	cells
				for (h = 0; h < thePuzzleSize; h++)
					for (v = 0; v < thePuzzleSize; v++)
						itsTextures[Texture2DWordSearchFirstCell + h*thePuzzleSize + v]
							= [self makeWordSearchCharacterMask:theBoard[h][v]];

				//	word marker
				itsTextures[Texture2DWordSearchMark]	= [theTextureLoader
															newTextureWithName:@"WordSearch/Mark"
															scaleFactor:1.0
															bundle:nil
															options:theTextureLoaderOptions
															error:&theError];

				//	hot spot
				itsTextures[Texture2DWordSearchHotSpot]	= [theTextureLoader
															newTextureWithName:@"WordSearch/HotSpot"
															scaleFactor:1.0
															bundle:nil
															options:theTextureLoaderOptions
															error:&theError];
			}
			break;
		
		case Game2DJigsaw:
			{
				itsTextures[Texture2DBackground]	= [self makeRGBATextureOfColor:
														(ColorP3Linear){1.000, 1.000, 0.918, 1.000} size:256];
				[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];

				id<MTLTexture>	theSourceImage,
								thePieceTemplate;
				
				//	theSourceImage should be in Display P3 coordinates
				//	(not extended-range sRGB coordinates) for the reason cited below.
				theSourceImage = [theTextureLoader
								newTextureWithName:[self jigsawPuzzleSourceImageNameWithModelData:md]
								scaleFactor:1.0
								bundle:nil
								options:theTextureLoaderOptions
								error:&theError];

				//	Note:  The piece template for iPad and iPad
				//	uses a slightly thicker black outline
				//	than the piece template for iPhone.
				thePieceTemplate = [theTextureLoader
								newTextureWithName:@"Jigsaw/PieceTemplate"
								scaleFactor:1.0
								bundle:nil
								options:theTextureLoaderOptions
								error:&theError];

				//	This call to -makeJigsawPieceCollagesWithSourceImage will initialize
				//
				//		itsTextures[Texture2DJigsawCollageWithoutBorders]
				//		itsTextures[Texture2DJigsawCollageWithBorders]
				//
				//	and will also set
				//
				//		md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter
				//		md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride
				//
				//	Note:  theSourceImage should be in Display P3 coordinates
				//	for the reason accompanying the parameter aSourceImage
				//	in makeJigsawPieceCollagesWithSourceImage.
				//
				[self makeJigsawPieceCollagesWithSourceImage:theSourceImage pieceTemplate:thePieceTemplate modelData:md];

				//	The MTLComputeCommandEncoder will have kept strong references
				//	to theSourceImage and thePieceTemplate, so it's OK to relinquish
				//	our own strong references.
			}
			break;
		
		case Game2DChess:

			itsTextures[Texture2DBackground]		= [theTextureLoader
														newTextureWithName:
#ifdef SHAPE_OF_SPACE_CHESSBOARD
															@"Chess/Board - parity B"
#else
															@"Chess/Board - parity A"
#endif
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessHalo]			= [theTextureLoader
														newTextureWithName:@"Chess/Halo"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessArrow]		= [theTextureLoader
														newTextureWithName:@"Chess/Arrow"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessTarget]		= [theTextureLoader
														newTextureWithName:@"Chess/Target"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessCheckmate]	= [theTextureLoader
														newTextureWithName:@"Chess/Checkmate"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessStalemate]	= [theTextureLoader
														newTextureWithName:@"Chess/Stalemate"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];


			itsTextures[Texture2DChessWhiteKing]	= [theTextureLoader
														newTextureWithName:@"Chess/King-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessWhiteQueen]	= [theTextureLoader
														newTextureWithName:@"Chess/Queen-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessWhiteBishop]	= [theTextureLoader
														newTextureWithName:@"Chess/Bishop-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessWhiteKnight]	= [theTextureLoader
														newTextureWithName:@"Chess/Knight-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessWhiteRook]	= [theTextureLoader
														newTextureWithName:@"Chess/Rook-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessWhitePawn]	= [theTextureLoader
														newTextureWithName:@"Chess/Pawn-wh"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];


			itsTextures[Texture2DChessBlackKing]	= [theTextureLoader
														newTextureWithName:@"Chess/King-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessBlackQueen]	= [theTextureLoader
														newTextureWithName:@"Chess/Queen-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessBlackBishop]	= [theTextureLoader
														newTextureWithName:@"Chess/Bishop-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessBlackKnight]	= [theTextureLoader
														newTextureWithName:@"Chess/Knight-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessBlackRook]	= [theTextureLoader
														newTextureWithName:@"Chess/Rook-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessBlackPawn]	= [theTextureLoader
														newTextureWithName:@"Chess/Pawn-bk"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];


			itsTextures[Texture2DChessDialBase]		= [theTextureLoader
														newTextureWithName:@"common/DialBase"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessDialSweep]	= [theTextureLoader
														newTextureWithName:@"common/DialSweep"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessDialRim]		= [theTextureLoader
														newTextureWithName:@"common/DialRim"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DChessDialArrow]	= [theTextureLoader
														newTextureWithName:@"common/DialArrow"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			break;
		
		case Game2DPool:

			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:
														(ColorP3Linear){0.0000, 0.6875, 0.2500, 1.0000} size:256];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];

			itsTextures[Texture2DPoolBall0]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball0"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall1]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball1"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall2]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball2"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall3]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball3"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall4]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball4"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall5]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball5"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall6]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball6"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolBall7]			= [theTextureLoader
														newTextureWithName:@"Pool/Ball7"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];


			itsTextures[Texture2DPoolPocket]		= [theTextureLoader
														newTextureWithName:@"Pool/Pocket"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolStick]			= [theTextureLoader
														newTextureWithName:@"Pool/Stick"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolLineOfSight]	= [theTextureLoader
														newTextureWithName:@"Pool/LineOfSight"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DPoolHotSpot]		= [theTextureLoader
														newTextureWithName:@"Pool/HotSpot"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			break;
		
		case Game2DApples:

			itsTextures[Texture2DBackground]		= [self makeRGBATextureOfColor:
														(ColorP3Linear){0.500, 0.750, 1.000, 1.000} size:256];
			[self roughenTexture:itsTextures[Texture2DBackground] roughingFactor:theRoughingFactor];

			itsTextures[Texture2DApplesApple]		= [theTextureLoader
														newTextureWithName:@"Apples/Apple"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DApplesWorm]		= [theTextureLoader
														newTextureWithName:@"Apples/Worm"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DApplesFaceHappy]	= [theTextureLoader
														newTextureWithName:@"Apples/FaceHappy"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			itsTextures[Texture2DApplesFaceYucky]	= [theTextureLoader
														newTextureWithName:@"Apples/FaceYucky"
														scaleFactor:1.0
														bundle:nil
														options:theTextureLoaderOptions
														error:&theError];

			{
				unsigned int	i;

				for (i = 1; i <= 8; i++)
					itsTextures[Texture2DApplesNumeralBaseIndex + i] = [self makeApplesNumeralMask:('0' + i)];
			}

			break;
		
		default:
			break;
	}
}

- (void)shutDownTextures
{
	unsigned int	i;
	
	for (i = 0; i < TotalNumTextures; i++)
		itsTextures[i] = nil;
}

- (void)refreshTexturesWithModelData:(ModelData *)md
{
	//	The game may have changed,
	//	so refresh all textures.

	[self shutDownTextures];
	[self setUpTexturesWithModelData:md];
}

- (void)refreshTexturesForGameResetWithModelData:(ModelData *)md
{
	//	When the user resets the current game,
	//	only a few textures need to get refreshed.
	
	switch (md->itsGame)
	{
		case Game2DMaze:
			itsTextures[Texture2DMazeMaze] = [self makeMazeMaskWithModelData:md];
			break;
		
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
			[self refreshTexturesWithModelData:(ModelData *)md];
			break;
			
		default:
			//	do nothing
			break;
	}
}

- (void)refreshTexturesForCharacterInputWithModelData:(ModelData *)md
{
	unsigned int	thePuzzleSize,
					h,
					v,
					theTextureIndex;
	Char16			(*theBoard)[MAX_CROSSWORD_SIZE],
					(*theSolution)[MAX_CROSSWORD_SIZE],
					thePendingCharacter;

	if (md->itsGame == Game2DCrossword)	//	should never fail
	{
		thePuzzleSize		= md->itsGameOf.Crossword2D.itsPuzzleSize;
		theBoard			= md->itsGameOf.Crossword2D.itsBoard;
		theSolution			= md->itsGameOf.Crossword2D.itsSolution;
		thePendingCharacter	= md->itsGameOf.Crossword2D.itsPendingCharacter;

		for (h = 0; h < thePuzzleSize; h++)
		{
			for (v = 0; v < thePuzzleSize; v++)
			{
				if (theSolution[h][v] != L'*')	//	not a black cell
				{
					theTextureIndex = Texture2DCrosswordFirstCell + h*thePuzzleSize + v;

					if (theBoard[h][v] != L' ')	//	not an empty cell
					{
						//	Update the texture only if the user has typed
						//	a new character into this cell.
						if (itsCrosswordTextureCharacters[h][v] != theBoard[h][v])	//	needs updating
						{
							itsTextures[theTextureIndex]		= [self makeCrosswordCharacterMask:theBoard[h][v]];
							itsCrosswordTextureCharacters[h][v]	= theBoard[h][v];
						}
					}
					else						//	empty cell
					{
						itsTextures[theTextureIndex]		= nil;
						itsCrosswordTextureCharacters[h][v]	= 0;
					}
				}
			}
		}

		//	Texture for the pending character (if any)
		//
		//		Technical note:  Don't bother checking whether the pending character
		//		has changed.  In practice almost any user input that leaves
		//		a non-trivial pending character will have modified the pending character
		//		from its previous value.
		//
		if (thePendingCharacter != NO_PENDING_CHARACTER)
			itsTextures[Texture2DCrosswordPendingCharacter] = [self makeCrosswordCharacterMask:thePendingCharacter];
		else
			itsTextures[Texture2DCrosswordPendingCharacter] = nil;
	}
}


- (void)setUpSamplers
{
	MTLSamplerDescriptor	*theDescriptor;
	
	//	It's unclear whether -newSamplerStateWithDescriptor: simply copies
	//	the data from theDescriptor, or whether it keeps a reference to the actual object.
	//	To be safe, let's start with a fresh MTLSamplerDescriptor for each sampler.
	
	
	theDescriptor = [[MTLSamplerDescriptor alloc] init];

	[theDescriptor setNormalizedCoordinates:YES];

	[theDescriptor setSAddressMode:MTLSamplerAddressModeRepeat];
	[theDescriptor setTAddressMode:MTLSamplerAddressModeRepeat];

	[theDescriptor setMinFilter:MTLSamplerMinMagFilterLinear];
	[theDescriptor setMagFilter:MTLSamplerMinMagFilterLinear];
	[theDescriptor setMipFilter:MTLSamplerMipFilterLinear];
	
	[theDescriptor setMaxAnisotropy:1];
	
	itsTextureSamplerIsotropicRepeat = [itsDevice newSamplerStateWithDescriptor:theDescriptor];

	
	theDescriptor = [[MTLSamplerDescriptor alloc] init];

	[theDescriptor setNormalizedCoordinates:YES];

	[theDescriptor setSAddressMode:MTLSamplerAddressModeClampToEdge];
	[theDescriptor setTAddressMode:MTLSamplerAddressModeClampToEdge];

	[theDescriptor setMinFilter:MTLSamplerMinMagFilterLinear];
	[theDescriptor setMagFilter:MTLSamplerMinMagFilterLinear];
	[theDescriptor setMipFilter:MTLSamplerMipFilterLinear];
	
	[theDescriptor setMaxAnisotropy:1];
	
	itsTextureSamplerIsotropicClampToEdge = [itsDevice newSamplerStateWithDescriptor:theDescriptor];
}

- (void)shutDownSamplers
{
	itsTextureSamplerIsotropicRepeat		= nil;
	itsTextureSamplerIsotropicClampToEdge	= nil;
}


#pragma mark -
#pragma mark textures

- (id<MTLTexture>)makeTicTacToeGridTextureWithColor:(ColorP3Linear)aColor		//	premultiplied alpha
											   size:(NSUInteger)aSize			//	power of two
										borderWidth:(NSUInteger)aBorderWidth	//	half the width of a grid line
{
	NSUInteger						theThreadExecutionWidth;
	MTLTextureDescriptor			*theDescriptor;
	id<MTLTexture>					theTexture;
	id<MTLCommandBuffer>			theCommandBuffer;
	id<MTLComputeCommandEncoder>	theComputeEncoder;
	simd_ushort2					theGridLineLimits;
#if TARGET_OS_IOS
	double							theLinearDisplayP3Color[3],
									theLinearXRsRGBColor[3];
#endif
	faux_simd_half4					theColor;
	NSUInteger						theThreadgroupWidth,
									theThreadgroupHeight;

	//	If the host doesn't support non-uniform threadgroup sizes,
	//	then to support our workaround we'll increase the texture size
	//	as necessary, to ensure that it's a multiple of the thread execution width.
	theThreadExecutionWidth = [itsMakeTicTacToeGridPipelineState threadExecutionWidth];
	if ( ! itsNonuniformThreadGroupSizesAreAvailable )
	{
		if (aSize % theThreadExecutionWidth != 0)
		{
			aSize = ( (aSize / theThreadExecutionWidth) + 1 ) * theThreadExecutionWidth;
		}
	}

	//	Create an empty texture.
	//
	//		Note:  MTLPixelFormatRGBA16Float supports
	//		both wide color and transparency on all platforms,
	//		as well as read-write (for -roughenTexture:)
	//		on newer hardware.  Given that Tic-Tac-Toe
	//		currently uses a non-saturated color for the grid lines,
	//		we could probably get by with a 32-bit pixel format
	//		like MTLPixelFormatBGRA8Unorm or MTLPixelFormatBGRA8Unorm_sRGB,
	//		but for now I'd rather keep things simple and robust.
	//
	theDescriptor = [MTLTextureDescriptor
						texture2DDescriptorWithPixelFormat:	MTLPixelFormatRGBA16Float
												width:		aSize
												height:		aSize
												mipmapped:	YES];
	[theDescriptor setUsage:(MTLTextureUsageShaderRead | MTLTextureUsageShaderWrite)];
	[theDescriptor setStorageMode:MTLStorageModePrivate];
	theTexture = [itsDevice newTextureWithDescriptor:theDescriptor];
	
	//	Set the location and width of the grid line, in pixels.
	//
	//		Note:  theGridLineLimits specify not pixels,
	//		but rather the coordinate lines between pixels.
	//		For example, to create the grid lines
	//		shown by the asterisks in this 16×16 texture
	//
	//		    * * * * * * * * * * * * * * * *
	//		    * * * * * * * * * * * * * * * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		    * *                         * *
	//		   |* *|* * * * * * * * * * * *|* *|
	//		   |* *|* * * * * * * * * * * *|* *|
	//		   0   2                       14  16
	//
	//		we'd pass theGridLineLimits = (2, 14).
	//
	theGridLineLimits = (simd_ushort2) {aBorderWidth, aSize - aBorderWidth};

	//	Set the color.
#if TARGET_OS_IOS
	//	On iOS, all rendering takes in place in extended-range linear sRGB coordinates.
	theLinearDisplayP3Color[0] = aColor.r;
	theLinearDisplayP3Color[1] = aColor.g;
	theLinearDisplayP3Color[2] = aColor.b;
	ConvertDisplayP3LinearToXRsRGBLinear(theLinearDisplayP3Color, theLinearXRsRGBColor);
	theColor	= (faux_simd_half4)
				{
					theLinearXRsRGBColor[0],
					theLinearXRsRGBColor[1],
					theLinearXRsRGBColor[2],
					aColor.a
				};
#else
	//	On macOS, all Geometry Games rendering takes place in linear Display P3.
	theColor = (faux_simd_half4) {aColor.r, aColor.g, aColor.b, aColor.a};
#endif

	//	Let the GPU draw Tac-Tac-Toe grid lines into theTexture.
	
	theCommandBuffer = [itsCommandQueue commandBuffer];
	
	theComputeEncoder = [theCommandBuffer computeCommandEncoder];
	[theComputeEncoder setLabel:@"make Tic-Tac-Toe grid"];
	[theComputeEncoder setComputePipelineState:itsMakeTicTacToeGridPipelineState];
	[theComputeEncoder setTexture:theTexture atIndex:TextureIndexCFImage];
	[theComputeEncoder setBytes:&theGridLineLimits length:sizeof(theGridLineLimits) atIndex:BufferIndexCFMisc];
	[theComputeEncoder setBytes:&theColor length:sizeof(faux_simd_half4) atIndex:BufferIndexCFColor];

	if (itsNonuniformThreadGroupSizesAreAvailable)
	{
		//	Dispatch one thread per pixel, using the first strategy
		//	described in Apple's article
		//
		//		https://developer.apple.com/documentation/metal/calculating_threadgroup_and_grid_sizes
		//
		//	There's no reason theThreadgroupWidth must equal the threadExecutionWidth,
		//	but it's a convenient choice.
		//
		theThreadgroupWidth  = [itsMakeTicTacToeGridPipelineState threadExecutionWidth];			//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = [itsMakeTicTacToeGridPipelineState maxTotalThreadsPerThreadgroup]	//	varies according to program resource needs
							 / theThreadgroupWidth;
		
		[theComputeEncoder dispatchThreads:	MTLSizeMake(aSize, aSize, 1)
					 threadsPerThreadgroup:	MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
	}
	else
	{
		//	Legacy method:
		//
		//	Use the second strategy described in Apple's article cited above.
		//
		//	We've already increased aWidth as needed to ensure
		//	that aSize is a multiple of the thread execution width.
		//	Thus by letting theThreadgroupHeight be 1, we guarantee
		//	that no threadgroup will extend beyond the bounds of the image.
		//	Therefore the compute function needn't include any "defensive code"
		//	to check for out-of-bounds pixel coordinates.
		//
		theThreadgroupWidth  = theThreadExecutionWidth;	//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = 1;

		[theComputeEncoder dispatchThreadgroups: MTLSizeMake(
													aSize / theThreadgroupWidth,
													aSize / theThreadgroupHeight,
													1)
						  threadsPerThreadgroup: MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
	}
	
	[theComputeEncoder endEncoding];

	[self generateMipmapsForTexture:theTexture commandBuffer:theCommandBuffer];

	[theCommandBuffer commit];

	return theTexture;
}

- (id<MTLTexture>)makeGomokuGridTextureWithColor:(ColorP3Linear)aColor		//	premultiplied alpha
											size:(NSUInteger)aSize			//	power of two
								   lineHalfWidth:(NSUInteger)aLineHalfWidth	//	half the width of a grid line

{
	NSUInteger						theThreadExecutionWidth;
	MTLTextureDescriptor			*theDescriptor;
	id<MTLTexture>					theTexture;
	id<MTLCommandBuffer>			theCommandBuffer;
	id<MTLComputeCommandEncoder>	theComputeEncoder;
	simd_ushort2					theGridLineLimits;
#if TARGET_OS_IOS
	double							theLinearDisplayP3Color[3],
									theLinearXRsRGBColor[3];
#endif
	faux_simd_half4					theColor;
	NSUInteger						theThreadgroupWidth,
									theThreadgroupHeight;

	//	If the host doesn't support non-uniform threadgroup sizes,
	//	then to support our workaround we'll increase the texture size
	//	as necessary, to ensure that it's a multiple of the thread execution width.
	theThreadExecutionWidth = [itsMakeGomokuGridPipelineState threadExecutionWidth];
	if ( ! itsNonuniformThreadGroupSizesAreAvailable )
	{
		if (aSize % theThreadExecutionWidth != 0)
		{
			aSize = ( (aSize / theThreadExecutionWidth) + 1 ) * theThreadExecutionWidth;
		}
	}

	//	Create an empty texture.
	//
	//		Note:  MTLPixelFormatRGBA16Float supports
	//		both wide color and transparency on all platforms,
	//		as well as read-write (for -roughenTexture:)
	//		on newer hardware.  Given that Gomoku
	//		currently uses a non-saturated color for the grid lines,
	//		we could probably get by with a 32-bit pixel format
	//		like MTLPixelFormatBGRA8Unorm or MTLPixelFormatBGRA8Unorm_sRGB,
	//		but for now I'd rather keep things simple and robust.
	//
	theDescriptor = [MTLTextureDescriptor
						texture2DDescriptorWithPixelFormat:	MTLPixelFormatRGBA16Float
												width:		aSize
												height:		aSize
												mipmapped:	YES];
	[theDescriptor setUsage:(MTLTextureUsageShaderRead | MTLTextureUsageShaderWrite)];
	[theDescriptor setStorageMode:MTLStorageModePrivate];
	theTexture = [itsDevice newTextureWithDescriptor:theDescriptor];
	
	//	Set the location and width of the grid line, in pixels.
	//
	//		Note:  theGridLineLimits specify not pixels,
	//		but rather the coordinate lines between pixels.
	//		For example, to create the grid lines
	//		shown by the asterisks in this 16×16 texture
	//
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    * * * * * * * * * * * * * * * *
	//		    * * * * * * * * * * * * * * * *
	//		    * * * * * * * * * * * * * * * *
	//		    * * * * * * * * * * * * * * * *
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		    · · · · · · * * * * · · · · · ·
	//		   |· · · · · ·|* * * *|· · · · · ·|
	//		   |· · · · · ·|* * * *|· · · · · ·|
	//		   0           6       10          16
	//
	//		we'd pass theGridLineLimits = (6, 10).
	//
	theGridLineLimits = (simd_ushort2) {
							aSize/2 - aLineHalfWidth,
							aSize/2 + aLineHalfWidth};

	//	Set the color.
#if TARGET_OS_IOS
	//	On iOS, all rendering takes in place in extended-range linear sRGB coordinates.
	theLinearDisplayP3Color[0] = aColor.r;
	theLinearDisplayP3Color[1] = aColor.g;
	theLinearDisplayP3Color[2] = aColor.b;
	ConvertDisplayP3LinearToXRsRGBLinear(theLinearDisplayP3Color, theLinearXRsRGBColor);
	theColor	= (faux_simd_half4)
				{
					theLinearXRsRGBColor[0],
					theLinearXRsRGBColor[1],
					theLinearXRsRGBColor[2],
					aColor.a
				};
#else
	//	On macOS, all Geometry Games rendering takes place in linear Display P3.
	theColor = (faux_simd_half4) {aColor.r, aColor.g, aColor.b, aColor.a};
#endif

	//	Let the GPU draw Tac-Tac-Toe grid lines into theTexture.
	
	theCommandBuffer = [itsCommandQueue commandBuffer];
	
	theComputeEncoder = [theCommandBuffer computeCommandEncoder];
	[theComputeEncoder setLabel:@"make Gomoku grid"];
	[theComputeEncoder setComputePipelineState:itsMakeGomokuGridPipelineState];
	[theComputeEncoder setTexture:theTexture atIndex:TextureIndexCFImage];
	[theComputeEncoder setBytes:&theGridLineLimits length:sizeof(theGridLineLimits) atIndex:BufferIndexCFMisc];
	[theComputeEncoder setBytes:&theColor length:sizeof(faux_simd_half4) atIndex:BufferIndexCFColor];

	if (itsNonuniformThreadGroupSizesAreAvailable)
	{
		//	Dispatch one thread per pixel, using the first strategy
		//	described in Apple's article
		//
		//		https://developer.apple.com/documentation/metal/calculating_threadgroup_and_grid_sizes
		//
		//	There's no reason theThreadgroupWidth must equal the threadExecutionWidth,
		//	but it's a convenient choice.
		//
		theThreadgroupWidth  = [itsMakeGomokuGridPipelineState threadExecutionWidth];			//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = [itsMakeGomokuGridPipelineState maxTotalThreadsPerThreadgroup]	//	varies according to program resource needs
							 / theThreadgroupWidth;
		
		[theComputeEncoder dispatchThreads:	MTLSizeMake(aSize, aSize, 1)
					 threadsPerThreadgroup:	MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
	}
	else
	{
		//	Legacy method:
		//
		//	Use the second strategy described in Apple's article cited above.
		//
		//	We've already increased aWidth as needed to ensure
		//	that aSize is a multiple of the thread execution width.
		//	Thus by letting theThreadgroupHeight be 1, we guarantee
		//	that no threadgroup will extend beyond the bounds of the image.
		//	Therefore the compute function needn't include any "defensive code"
		//	to check for out-of-bounds pixel coordinates.
		//
		theThreadgroupWidth  = theThreadExecutionWidth;	//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = 1;

		[theComputeEncoder dispatchThreadgroups: MTLSizeMake(
													aSize / theThreadgroupWidth,
													aSize / theThreadgroupHeight,
													1)
						  threadsPerThreadgroup: MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
	}
	
	[theComputeEncoder endEncoding];

	[self generateMipmapsForTexture:theTexture commandBuffer:theCommandBuffer];

	[theCommandBuffer commit];

	return theTexture;
}

- (NSString *)jigsawPuzzleSourceImageNameWithModelData:(ModelData *)md
{
	static NSString	*theTorusPuzzleFileNames[] =
					{
						@"Jigsaw/Torus/Flowers",
						@"Jigsaw/Torus/Parrots",
						@"Jigsaw/Torus/Sheep",
						@"Jigsaw/Torus/Balloons",
						@"Jigsaw/Torus/Tropical Fruit",
						@"Jigsaw/Torus/Physics",
						@"Jigsaw/Torus/Strawberries",
						@"Jigsaw/Torus/Math",
						@"Jigsaw/Torus/Vegetables",
						@"Jigsaw/Torus/Citrus",
						@"Jigsaw/Torus/Chemistry"
					},
					*theKleinPuzzleFileNames[] =
					{
						@"Jigsaw/Klein Bottle/Home",
						@"Jigsaw/Klein Bottle/Vines",
						@"Jigsaw/Klein Bottle/Balloons - Klein Bottle",
						@"Jigsaw/Klein Bottle/Sheep - Klein Bottle"
					};

	GEOMETRY_GAMES_ASSERT(
		BUFFER_LENGTH(theTorusPuzzleFileNames) == NUM_TORUS_PUZZLE_IMAGES,
		"Wrong number of torus puzzle file names");

	GEOMETRY_GAMES_ASSERT(
		BUFFER_LENGTH(theKleinPuzzleFileNames) == NUM_KLEIN_PUZZLE_IMAGES,
		"Wrong number of Klein bottle puzzle file names");

	//	The puzzle image depends on the current topology and puzzle number.
	switch (md->itsTopology)
	{
		case Topology2DTorus:

			GEOMETRY_GAMES_ASSERT(
				md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle < NUM_TORUS_PUZZLE_IMAGES,
				"itsCurrentTorusPuzzle is invalid");
			
			return theTorusPuzzleFileNames[md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle];

		case Topology2DKlein:

			GEOMETRY_GAMES_ASSERT(
				md->itsGameOf.Jigsaw2D.itsCurrentKleinPuzzle < NUM_KLEIN_PUZZLE_IMAGES,
				"itsCurrentKleinPuzzle is invalid");

			return theKleinPuzzleFileNames[md->itsGameOf.Jigsaw2D.itsCurrentKleinPuzzle];
		
		default:

			GEOMETRY_GAMES_ABORT("Invalid topology in jigsawPuzzleSourceImageNameWithModelData");

			return NULL;	//	suppress compiler warnings
	}
}

- (void)makeJigsawPieceCollagesWithSourceImage:(id<MTLTexture>)aSourceImage
				//	aSourceImage should use non-extended color coordinates.
				//	TorusGamesComputeFunctionMakeJigsawCollages() will interpret
				//	them as Display P3 and (on iOS) convert them
				//	to extended-range sRGB coordinates as required.
				//	As the comment accompanying the use of
				//	theBorderlessCollagePixelColorAsRescaledP3
				//	in TorusGamesComputeFunctionMakeJigsawCollages()
				//	explains, our motivation for avoiding extended-range
				//	color coordinates in aSourceImage is to take advantage
				//	of ASTC texture compression in our Asset Catalog.
								 pieceTemplate:(id<MTLTexture>)aPieceTemplate
									 modelData:(ModelData *)md
{
	NSUInteger						theSourceSizePx,	//	source image's width and height in pixels
									theCollageSizePx;	//	collage's width and height in pixels
	MTLTextureDescriptor			*theDescriptorForTextureWithoutBorders,
									*theDescriptorForTextureWithBorders;
	JigsawCollageLayout				theCollageLayout;
	id<MTLCommandBuffer>			theCommandBuffer;
	id<MTLComputeCommandEncoder>	theComputeEncoder;
	NSUInteger						theThreadgroupWidth,
									theThreadgroupHeight;

	//	If the raw image file couldn't be read, substitute a pure yellow image
	//	so that the app can keep running while still letting the user know
	//	that something went wrong.  The yellow image must be large enough
	//	that the corresponding collage, at twice the yellow image's size,
	//	will have enough room for each piece to have its own region,
	//	to accommodate its own black border curves.  Moreover, those border curves
	//	will look better if the yellow texture has the same resolution
	//	as the standard textures.
	if (aSourceImage == nil)
	{
		aSourceImage = [self makeRGBATextureOfColor:(ColorP3Linear){1.0, 1.0, 0.0, 1.0} size:2048];
	}

	//	The puzzle image must be square.
	GEOMETRY_GAMES_ASSERT(	[aSourceImage width] == [aSourceImage height],
							"Jigsaw puzzle image must be square");
	theSourceSizePx  = [aSourceImage width];
	theCollageSizePx = 2 * theSourceSizePx;


	//	Create the collage textures.
	
	theDescriptorForTextureWithoutBorders = [MTLTextureDescriptor
		texture2DDescriptorWithPixelFormat:	MTLPixelFormatRGBA16Float //	Supports both wide color and (unnecessarily) transparency
								width:		theCollageSizePx
								height:		theCollageSizePx
								mipmapped:	NO];	//	Collage without borders is never displayed on screen
	[theDescriptorForTextureWithoutBorders
		setUsage: MTLTextureUsageShaderRead		//	Needs read access and write access at different times,
				| MTLTextureUsageShaderWrite];	//		but doesn't need simultaneous read_write access.
	[theDescriptorForTextureWithoutBorders setStorageMode:MTLStorageModePrivate];
	itsTextures[Texture2DJigsawCollageWithoutBorders] = [itsDevice newTextureWithDescriptor:theDescriptorForTextureWithoutBorders];
	
	theDescriptorForTextureWithBorders = [MTLTextureDescriptor
		texture2DDescriptorWithPixelFormat:	MTLPixelFormatRGBA16Float //	Supports both wide color and transparency
								width:		theCollageSizePx
								height:		theCollageSizePx
								mipmapped:	YES];	//	Collage with borders does get displayed on screen
	[theDescriptorForTextureWithBorders
		setUsage: MTLTextureUsageShaderRead		//	Needs read access and write access at different times,
				| MTLTextureUsageShaderWrite];	//		but doesn't need simultaneous read_write access.
	[theDescriptorForTextureWithBorders setStorageMode:MTLStorageModePrivate];
	itsTextures[Texture2DJigsawCollageWithBorders] = [itsDevice newTextureWithDescriptor:theDescriptorForTextureWithBorders];


	//	Compute the collage's layout parameters.
	//
	//		Note:  The method
	//			computeJigsawCollageLayout:rawImageSize:modelData:
	//		also sets
	//			md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter
	//			md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride
	//
	[self computeJigsawCollageLayout:&theCollageLayout
					 sourceImageSize:theSourceSizePx
				  pieceTemplateWidth:[aPieceTemplate width]
				 pieceTemplateHeight:[aPieceTemplate height]
						   modelData:md];


	theCommandBuffer = [itsCommandQueue commandBuffer];
	
	theComputeEncoder = [theCommandBuffer computeCommandEncoder];
	[theComputeEncoder setLabel:@"make Jigsaw collage"];
	[theComputeEncoder setComputePipelineState:itsMakeJigsawCollagesPipelineState];

	[theComputeEncoder setTexture:aSourceImage
						atIndex:TextureIndexCFJigsawPuzzleSourceImage];

	[theComputeEncoder setTexture:aPieceTemplate
						atIndex:TextureIndexCFJigsawPieceTemplate];

	[theComputeEncoder setTexture:itsTextures[Texture2DJigsawCollageWithoutBorders]
						atIndex:TextureIndexCFJigsawCollageWithoutBorders];

	[theComputeEncoder setTexture:itsTextures[Texture2DJigsawCollageWithBorders]
						atIndex:TextureIndexCFJigsawCollageWithBorders];

	[theComputeEncoder setBytes:&theCollageLayout length:sizeof(theCollageLayout) atIndex:BufferIndexCFMisc];

	if (itsNonuniformThreadGroupSizesAreAvailable)
	{
		//	Dispatch one thread per pixel, using the first strategy
		//	described in Apple's article
		//
		//		https://developer.apple.com/documentation/metal/calculating_threadgroup_and_grid_sizes
		//
		//	There's no reason theThreadgroupWidth must equal the threadExecutionWidth,
		//	but it's a convenient choice.
		//
		theThreadgroupWidth  = [itsMakeJigsawCollagesPipelineState threadExecutionWidth];			//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = [itsMakeJigsawCollagesPipelineState maxTotalThreadsPerThreadgroup]	//	varies according to program resource needs
							 / theThreadgroupWidth;
		
		[theComputeEncoder dispatchThreads:	MTLSizeMake(theCollageSizePx, theCollageSizePx, 1)
					 threadsPerThreadgroup:	MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
	}
	else
	{
		//	Legacy method:
		//
		//	Use the second strategy described in Apple's article
		//
		//		https://developer.apple.com/documentation/metal/calculating_threadgroup_and_grid_sizes
		//
		//	The threadExecutionWidth is typically a fairly small power of two
		//	(perhaps 2⁵ = 32), while our puzzles image sizes are all larger
		//	powers of two (2¹⁰ = 1024 on iPhoneOS or 2¹¹ = 2048 on iPadOS or macOS),
		//	so we fully expect the image width to be an exact multiple
		//	of the threadExecutionWidth.  If we let theThreadgroupHeight be 1,
		//	then we ensure that no threadgroup will extend beyond the bounds of the image.
		//	In other words, all threads will correspond to valid pixels,
		//	and the compute function needn't include any "defensive code"
		//	to check for out-of-bounds pixel coordinates.
		//
		theThreadgroupWidth  = [itsMakeJigsawCollagesPipelineState threadExecutionWidth];	//	hardware-dependent constant (typically 32)
		theThreadgroupHeight = 1;

		if (theCollageSizePx % theThreadgroupWidth == 0		//	Always the case with current puzzle images
		 && theCollageSizePx % theThreadgroupHeight == 0)	//	Always the case
		{
			[theComputeEncoder dispatchThreadgroups: MTLSizeMake(
														theCollageSizePx / theThreadgroupWidth,
														theCollageSizePx / theThreadgroupHeight,
														1)
							  threadsPerThreadgroup: MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
		}
		else	//	Never the case with current puzzle images
		{
			//	Substitute a pure cyan texture, so the app may continue
			//	while still giving some indication that there's a problem.
			itsTextures[Texture2DJigsawCollageWithoutBorders] =
				[self makeRGBATextureOfColor:(ColorP3Linear){0.0, 1.0, 1.0, 1.0}	//	cyan
												size:(unsigned int)theCollageSizePx];
			itsTextures[Texture2DJigsawCollageWithBorders] =
				[self makeRGBATextureOfColor:(ColorP3Linear){0.0, 1.0, 1.0, 1.0}	//	cyan
												size:(unsigned int)theCollageSizePx];
		}
	}
	
	[theComputeEncoder endEncoding];

	//	The borderless collage is never displayed directly, so no need for mipmaps.
	//	The bordered collage, on the other hand, does need mipmaps.
	[self generateMipmapsForTexture:itsTextures[Texture2DJigsawCollageWithBorders] commandBuffer:theCommandBuffer];

	[theCommandBuffer commit];
}

- (void)computeJigsawCollageLayout:(JigsawCollageLayout *)aCollageLayout
				   sourceImageSize:(NSUInteger)aSourceSizePx
				pieceTemplateWidth:(NSUInteger)aPieceTemplateWidthPx
			   pieceTemplateHeight:(NSUInteger)aPieceTemplateHeightPx
						 modelData:(ModelData *)md
{
	NSUInteger		n,
					s,
					theCoordinateMask;
	TopologyType	thePuzzleTopology;
	NSUInteger		thePhaseShiftHPx,	//	horizontal phase shift in pixels, ∈ {0, …, s-1}
					thePhaseShiftVPx,	//	 vertical  phase shift in pixels, ∈ {0, …, s-1}
					theSubsequentOffsetPx,
					theInitialOffsetPx,
					theApproxBlockStride;
	double			thePieceWidthsPerPixel;
	NSUInteger		i;
	JigsawPiece		*thePiece;
	

	//	Purpose
	//
	//	Given an s × s source image, the collage will be a single 2s × 2s image
	//	containing each puzzle piece in its own separate block,
	//	with plenty of room for any "tabs" that may extend outward from it.
	//	The present method computes the layout parameters for the collage.

	//	Notation
	//
	//	The following code works with fractions whose denominators
	//	are various combinations of
	//
	//		 n = the puzzle size
	//				that is, the number of puzzle pieces in each direction,
	//				typically 3, 4 or 5,
	//
	//		 s = the source image size,
	//				that is, the width and height of original image, in pixels,
	//				typically 1024 or 2048, and
	//
	//		2s = the collage size,
	//				that is, the width and height of the collage, in pixels,
	//				typically 2048 or 4096.
	//
	//	For each such variable, a suffix denotes the denominator,
	//	for example the suffix "2sn" denotes a denominator 2*s*n.
	//	Because the denominator of all such fractions is known,
	//	it suffices to record only their numerators.
	
	//	The following parameters are given.
	n					= md->itsGameOf.Jigsaw2D.itsSize;
	s					= aSourceSizePx;
	thePuzzleTopology	= md->itsTopology;
	
	//	Set random phase shifts.
	//
	//		Note:  Even though we expect s to be a power of two (for easy mipmapping),
	//		the following code relies only on s being even.
	//
	GEOMETRY_GAMES_ASSERT(s % 2 == 0, "The puzzle image must have even width.");
	if (thePuzzleTopology == Topology2DKlein)
		thePhaseShiftHPx = ( RandomBoolean() ? s/2 : 0 );
	else
		thePhaseShiftHPx = RandomUnsignedInteger() % s;
	thePhaseShiftVPx = RandomUnsignedInteger() % s;
	
	//	The raw size s should always be a power of two...
	GEOMETRY_GAMES_ASSERT(IsPowerOfTwo((unsigned int)s), "Jigsaw puzzle image size must be a power of two");
	
	//	...so one convenient corollary of this is that we can
	//	wrap pixel coordinates by masking with s - 1
	//	(with an appropriate adjustment for the flip
	//	in a Klein bottle image).  For example, if
	//
	//			s = 1024 = 0x0400
	//	then
	//			theCoordinateMask = s - 1 = 0x03FF = 0000001111111111
	//
	theCoordinateMask = s - 1;
	
	//	Layout
	//
	//	In very rough terms, the source image consists of an n × n array of puzzle pieces,
	//	interconnected by tabs and slots.  Each puzzle piece is s/n pixels wide and s/n pixels high,
	//	excluding its tabs, but keep in mind that s/n is typically not an integer.
	//
	//	Our plan, still speaking in rough terms, is to create a 2s × 2s image
	//	that provides a 2s/n × 2s/n square for each puzzle piece, with the piece itself
	//	centered in the 2s/n × 2s/n square and with its tabs extending into the "extra" space.
	//	By allowing a 2s/n × 2s/n block for each s/n × s/n piece, we can draw all necessary tabs
	//	without interfering with neighboring pieces or their tabs.
	//
	//	The tricky part is that in a typical case, say a 1024 × 1024 pixel image divided
	//	into a 3 × 3 array of puzzle pieces, the individual pieces do not align
	//	with the integer pixel grid.  In other words, the pieces may contain fractional pixels
	//	along their boundaries.  This is OK, just so we make sure that a given piece
	//	bears the same relation to the pixel grid in the new 2s × 2s collage that it did
	//	in the original s × s image.  This requirement will be realized if we ensure
	//	that each piece in the 2s × 2s collage sits at some integer offset
	//	from that piece's coordinates in the original s × s source image.
	//
	//	For ease of illustration, consider an 8 × 8 pixel source image:
	//
	//		  0  1  2  3  4  5  6  7  8
	//		0 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		1 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		2 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		3 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		4 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		5 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		6 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		7 +--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|
	//		8 +--+--+--+--+--+--+--+--+
	//
	//	When we cut the 8 × 8 source image into a 3 × 3 array of puzzle pieces,
	//	the cut lines (with horizontal and vertical coordinates 8/3 and 16/3)
	//	will slice through some pixels' interiors.
	//
	//		  0  1  2  3  4  5  6  7  8
	//		0 #########################
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		1 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		2 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #########################
	//		3 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		4 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		5 #--+--+-#+--+--+#-+--+--#
	//		  #########################
	//		  #··|··|·#|··|··|#·|··|··#
	//		6 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		7 #--+--+-#+--+--+#-+--+--#
	//		  #··|··|·#|··|··|#·|··|··#
	//		  #··|··|·#|··|··|#·|··|··#
	//		8 #########################
	//
	//	The collage needs to allow room for each piece's slots and tabs,
	//	so it offsets each piece by some integer amount
	//	relative to where the piece sits in the original source image (immediately above).
	//	Because the offset in each direction is always an integer,
	//	each piece bears the same relation to the integer pixel grid in the collage
	//	as it did to the integer pixel grid in the original source image.
	//
	//		  0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
	//		0 +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		1 +--#########+--+-#########-+--+#########--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		2 +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		3 +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#########|··|·#########·|··|#########··|··|··|
	//		4 +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		5 +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··#########|··|·#########·|··|#########··|··|··|
	//		6 +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		7 +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		8 +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#########|··|·#########·|··|#########··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		9 +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		A +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··#########|··|·#########·|··|#########··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		B +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		C +--#--+--+-#+--+-#+--+--+#-+--+#-+--+--#--+--+--+
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		  |··#··|··|·#|··|·#|··|··|#·|··|#·|··|··#··|··|··|
	//		D +--#########+--+-#########-+--+#########--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		E +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		F +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  |··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|··|
	//		  +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
	//
	//	In this example, the first piece in each row (resp. column) of the collage
	//	is offset by 1 pixel horizontally (resp. vertically) from that piece's
	//	position in the original source image, the second piece is offset
	//	by 3 pixels, the third piece by 5 pixels.
	//
	//	In general, each piece in column (resp. row) n of the collage
	//	is offset horizontally (resp. vertically) by
	//
	//		theInitialOffsetPx  +  n * theSubsequentOffsetPx
	//
	//	pixels, where theInitialOffsetPx and theSubsequentOffsetPx are defined as follows:
	//
	theSubsequentOffsetPx	= s / n;						//	fractional part intentionally discarded
	theInitialOffsetPx		= theSubsequentOffsetPx / 2;	//	fractional part intentionally discarded
	
	//	Error analysis
	//
	//	The integer value of theSubsequentOffsetPx differs
	//		from the exact fraction s/n by less than 1 pixel, and
	//	the integer value of theInitialOffsetPx differs
	//		from the exact fraction s/(2n) by less than 1 pixel,
	//	so even the last block in each row or column
	//	is offset from its "exact position" by less than n pixels.
	//	Given that in practice n is much smaller than s,
	//	our GPU compute functions may assign collage pixels
	//	to puzzle pieces in the simplest possible way,
	//	without worrying about whether one block might get
	//	a couple more or fewer pixels at the expense of its neighbor.

	//	The exact "block stride" in the collage,
	//	that is, the distance from the center of one piece
	//	to the center of the next, isn't an integer.
	//	Nevertheless, an integer approximation to the block stride
	//	will be useful in the GPU code when deciding which block
	//	a given pixel belongs to.  We needn't worry about pixels
	//	near the boundary between two blocks, because such pixels
	//	never get drawn, no matter which of the nearby blocks
	//	we assign them to.
	//
	//		Note:  The (n-1) term serves to "round up"
	//		in the integer division.  Rounding up ensures
	//		that the GPU function
	//
	//			TorusGamesComputeFunctionMakeJigsawCollages()
	//
	//		will always get a valid value for theBlock.
	//
	theApproxBlockStride = (2*s + (n-1)) / n;	//	fractional part intentionally discarded

	//	The GPU code first computes a pixel's exact location
	//	in the source image in pixel coordinates ∈ {0, …, s-1} × {0, …, s-1}.
	//	It then must switch to "piece coordinates",
	//	in which each piece has width 1.0, in order
	//	to determine where the pixel sits relative
	//	to the piece's curved border.  Given that n pieces
	//	fill a width (or height) of s pixels, the required
	//	conversion factor is simply n/s
	//	(as a floating-point number, not an integer).
	//
	thePieceWidthsPerPixel = (double)n / (double)s;
	
	//	This renderer and the C code both see the #definition
	//		of MAX_JIGSAW_SIZE in TorusGames-Common.h, while
	//	this renderer and the GPU code both see the #definition
	//		of COLLAGE_MAX_JIGSAW_SIZE in TorusGamesGPUDefinitions.h.
	//	But it's a bit awkward trying to find a way let all the code
	//		see a single unified #definition, short of creating
	//		a new file for such a #definition to live in
	//		(or letting either the GPU code or the C code see
	//		all sorts of inclusions that they really shouldn't see).
	//	So for now let's just stick with two separate definitions (ugh)
	//		and check that they agree.
#if (COLLAGE_MAX_JIGSAW_SIZE != MAX_JIGSAW_SIZE)
#error inconsistent Jigsaw collage sizes
#endif
	
	//	Copy our results into the JigsawCollageLayout for use on the GPU.
	aCollageLayout->n						= (ushort) n;
	aCollageLayout->s						= (ushort) s;
	aCollageLayout->itsCoordinateMask		= (ushort) theCoordinateMask;
	aCollageLayout->itsInitialOffsetPx		= (ushort) theInitialOffsetPx;
	aCollageLayout->itsSubsequentOffsetPx	= (ushort) theSubsequentOffsetPx;
	aCollageLayout->itsApproxBlockStride	= (ushort) theApproxBlockStride;
	aCollageLayout->itsPhaseShift			= (simd_ushort2) {thePhaseShiftHPx, thePhaseShiftVPx};
	aCollageLayout->itsKleinBottleFlag		= (thePuzzleTopology == Topology2DKlein);
	aCollageLayout->itsPieceWidthsPerPixel	= (float) thePieceWidthsPerPixel;
	aCollageLayout->itsPieceTemplateSize	= (simd_ushort2) {aPieceTemplateWidthPx, aPieceTemplateHeightPx};
	for (i = 0; i < n*n; i++)
	{
		thePiece = &md->itsGameOf.Jigsaw2D.itsPieces[i];
		aCollageLayout->itsPieceTabs
				[thePiece->itsLogicalH]
				[thePiece->itsLogicalV]
			= (simd_uchar4)
			{
				thePiece->itsTabGenderWest,
				thePiece->itsTabGenderEast,
				thePiece->itsTabGenderSouth,
				thePiece->itsTabGenderNorth
			};
	}

	//	On the CPU side, we'll need to know where
	//	the puzzle pieces' centers sit in the collage,
	//	so that we may assign texture coordinate correctly.

	md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter	//	in [0.0, 2.0] coordinates
		= 0.5 / (double)n
		+ (double)theInitialOffsetPx / (double)s;

	md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride		//	in [0.0, 2.0] coordinates
		= 1.0 / (double)n
		+ (double)theSubsequentOffsetPx / (double)s;
}


#pragma mark -
#pragma mark meshes

- (MeshBufferPair *)makeCube
{
	MeshBufferPair	*theMeshBufferPair;

	theMeshBufferPair = [[MeshBufferPair alloc] init];

	theMeshBufferPair->itsNumFaces = gCubeNumFacets;

	theMeshBufferPair->itsVertexBuffer = [itsDevice
		newBufferWithBytes:	gCubeVertices
		length:				sizeof(gCubeVertices)
		options:			MTLResourceStorageModeShared];

	theMeshBufferPair->itsIndexBuffer = [itsDevice
		newBufferWithBytes:	gCubeFacets
		length:				sizeof(gCubeFacets)
		options:			MTLResourceStorageModeShared];
	
	return theMeshBufferPair;
}

- (MeshBufferPair *)makeCubeSkeletonOuterFaces
{
	MeshBufferPair	*theMeshBufferPair;

	theMeshBufferPair = [[MeshBufferPair alloc] init];

	theMeshBufferPair->itsNumFaces = gCubeSkeletonNumFacets;

	theMeshBufferPair->itsVertexBuffer = [itsDevice
		newBufferWithBytes:	gCubeSkeletonVertices
		length:				sizeof(gCubeSkeletonVertices)
		options:			MTLResourceStorageModeShared];

	theMeshBufferPair->itsIndexBuffer = [itsDevice
		newBufferWithBytes:	gCubeSkeletonFacets
		length:				sizeof(gCubeSkeletonFacets)
		options:			MTLResourceStorageModeShared];
	
	return theMeshBufferPair;
}

- (MeshBufferPair *)makeCubeSkeletonInnerFaces
{
	MeshBufferPair						*theMeshBufferPair;
	unsigned int						i,
										j;
	TorusGames3DPolyhedronVertexData	theCubeSkeletonInnerVertices[NUM_CUBE_SKELETON_VERTICES];
	unsigned short						theCubeSkeletonInnerFacets[NUM_CUBE_SKELETON_FACETS][3];

	theMeshBufferPair = [[MeshBufferPair alloc] init];

	//	There are, of course, the same number of inner faces as outer faces.
	//
	theMeshBufferPair->itsNumFaces = gCubeSkeletonNumFacets;
	
	//	The inner vertices' positions are the same as the outer vertices' positions,
	//	but their normal vectors point in opposite directions.
	//
	for (i = 0; i < NUM_CUBE_SKELETON_VERTICES; i++)
	{
		theCubeSkeletonInnerVertices[i] = gCubeSkeletonVertices[i];
		
		for (j = 0; j < 3; j++)
			theCubeSkeletonInnerVertices[i].nor[j] *= -1.0;
	}
	theMeshBufferPair->itsVertexBuffer = [itsDevice
		newBufferWithBytes:	theCubeSkeletonInnerVertices
		length:				sizeof(theCubeSkeletonInnerVertices)
		options:			MTLResourceStorageModeShared];

	//	The inner facets are the same as the outer facets,
	//	but with opposite winding direction.
	//
	for (i = 0; i < NUM_CUBE_SKELETON_FACETS; i++)
	{
		theCubeSkeletonInnerFacets[i][0] = gCubeSkeletonFacets[i][0];
		theCubeSkeletonInnerFacets[i][1] = gCubeSkeletonFacets[i][2];	//	swap index 1 and index 2
		theCubeSkeletonInnerFacets[i][2] = gCubeSkeletonFacets[i][1];
	}
	theMeshBufferPair->itsIndexBuffer = [itsDevice
		newBufferWithBytes:	theCubeSkeletonInnerFacets
		length:				sizeof(theCubeSkeletonInnerFacets)
		options:			MTLResourceStorageModeShared];
	
	return theMeshBufferPair;
}

- (MeshBufferPair *)makeBallWithRefinementLevel:(unsigned int)aRefinementLevel
{
	MeshBufferPair	*theMeshBufferPair;

	GEOMETRY_GAMES_ASSERT(
		aRefinementLevel <= MAX_BALL_REFINEMENT_LEVEL,
		"aRefinementLevel exceeds the highest level that InitBallMeshes() prepares");
	
	InitBallMeshes();	//	one-time initialization

	theMeshBufferPair = [[MeshBufferPair alloc] init];

	theMeshBufferPair->itsNumFaces = gBallNumFacets[aRefinementLevel];

	theMeshBufferPair->itsVertexBuffer = [itsDevice
		newBufferWithBytes:	gBallVertices[aRefinementLevel]
		length:				gBallNumVertices[aRefinementLevel] * sizeof(TorusGames3DPolyhedronVertexData)
		options:			MTLResourceStorageModeShared];

	theMeshBufferPair->itsIndexBuffer = [itsDevice
		newBufferWithBytes:	gBallFacets[aRefinementLevel]
		length:				gBallNumFacets[aRefinementLevel] * sizeof(unsigned short [3])
		options:			MTLResourceStorageModeShared];
	
	return theMeshBufferPair;
}

- (MeshBufferPair *)makeTubeWithRefinementLevel:(unsigned int)aRefinementLevel
{
	MeshBufferPair	*theMeshBufferPair;

	GEOMETRY_GAMES_ASSERT(
		aRefinementLevel <= MAX_TUBE_REFINEMENT_LEVEL,
		"aRefinementLevel exceeds the highest level that InitTubeMeshes() prepares");
	
	InitTubeMeshes();	//	one-time initialization

	theMeshBufferPair = [[MeshBufferPair alloc] init];

	theMeshBufferPair->itsNumFaces = gTubeNumFacets[aRefinementLevel];

	theMeshBufferPair->itsVertexBuffer = [itsDevice
		newBufferWithBytes:	gTubeVertices[aRefinementLevel]
		length:				gTubeNumVertices[aRefinementLevel] * sizeof(TorusGames3DPolyhedronVertexData)
		options:			MTLResourceStorageModeShared];

	theMeshBufferPair->itsIndexBuffer = [itsDevice
		newBufferWithBytes:	gTubeFacets[aRefinementLevel]
		length:				gTubeNumFacets[aRefinementLevel] * sizeof(unsigned short [3])
		options:			MTLResourceStorageModeShared];
	
	return theMeshBufferPair;
}

- (id<MTLBuffer>)makeCircularSliceWithRefinementLevel:(unsigned int)aRefinementLevel
{
	id<MTLBuffer>	theCircularSliceTriangleStripBuffer;
	
	GEOMETRY_GAMES_ASSERT(
		aRefinementLevel <= MAX_SLICE_REFINEMENT_LEVEL,
		"aRefinementLevel exceeds the highest level that InitCircularSliceTriangleStrips() prepares");
	
	InitCircularSliceTriangleStrips();	//	one-time initialization

	theCircularSliceTriangleStripBuffer = [itsDevice
		newBufferWithBytes:	gCircularSliceVertices[aRefinementLevel]
		length:				gCircularSliceNumVertices[aRefinementLevel] * sizeof(TorusGames3DPolyhedronVertexData)
		options:			MTLResourceStorageModeShared];
	
	return theCircularSliceTriangleStripBuffer;
}


#pragma mark -
#pragma mark render

- (NSDictionary<NSString *, id> *)prepareInflightDataBuffersAtIndex:(unsigned int)anInflightBufferIndex modelData:(ModelData *)md
{
	NSMutableDictionary<NSString *, id<MTLBuffer>>	*theDictionary;
	TransformationBufferPair						*theTransformationBufferPair;

	//	On the one hand, if the user isn't dragging anything and
	//	no simulations are running, then we don't really
	//	need to update the instance data every frame.
	//	On the other hand, the updating uses a negligible amount
	//	of CPU time per frame (it's writing into a pre-allocated buffer),
	//	so I think it's better to go ahead and update every frame,
	//	to avoid cluttering up the app with code to keep track
	//	of whether the instance data does or doesn't need an update.

	//	Note:  The method -setValue:forKey: automatically suppresses nil values,
	//	unlike -setObject:forKey: .  But for now, I don't think Torus Games
	//	will ever encounter nil values here.

	theDictionary = [[NSMutableDictionary<NSString *, id<MTLBuffer>> alloc] initWithCapacity:16];

	if (GameIs3D(md->itsGame))
	{
		//	Release any previously allocated 2D buffers.
		its2DUniformBuffers[anInflightBufferIndex]					= nil;
		its2DCoveringTransformationBuffers[anInflightBufferIndex]	= nil;
		its2DSpritePlacementBuffers[anInflightBufferIndex]			= nil;
		
		//	Write current data into the 3D buffers.
		//	The write3D…IntoBuffer methods typically
		//	re-use the existing buffers that we pass into them.
		//	Only in the exceptional case that the required buffer size
		//	has changed will the write2D…IntoBuffer method replace
		//	the passed-in buffer with a new one of the correct size.

		//		uniform buffer
		
		its3DUniformBuffers[anInflightBufferIndex]
			= [self write3DUniformsIntoBuffer:its3DUniformBuffers[anInflightBufferIndex] modelData:md];

		[theDictionary setValue:	its3DUniformBuffers[anInflightBufferIndex]
						 forKey:	@"3D uniform buffer"];

		//		covering transformation buffers
		
		theTransformationBufferPair
			= [self write3DCoveringTransformationsIntoBuffersPlain:	its3DPlainCoveringTransformationBuffers[anInflightBufferIndex]
														reflecting:	its3DReflectingCoveringTransformationBuffers[anInflightBufferIndex]
														 modelData:	md];

		its3DPlainCoveringTransformationBuffers[anInflightBufferIndex]		= theTransformationBufferPair->itsPlainTransformations;
		its3DReflectingCoveringTransformationBuffers[anInflightBufferIndex]	= theTransformationBufferPair->itsReflectingTransformations;

		[theDictionary setValue:	theTransformationBufferPair->itsPlainTransformations
						 forKey:	@"3D plain covering transformation buffer"		];
		[theDictionary setValue:	theTransformationBufferPair->itsReflectingTransformations
						 forKey:	@"3D reflecting covering transformation buffer"	];

		//		polyhedron placement buffer
		
		its3DPolyhedronPlacementBuffers[anInflightBufferIndex]
			= [self write3DPolyhedronPlacementsIntoBuffer:its3DPolyhedronPlacementBuffers[anInflightBufferIndex] modelData:md];

		[theDictionary setValue:	its3DPolyhedronPlacementBuffers[anInflightBufferIndex]	//	Clears the key if
						 forKey:	@"3D polyhedron placement buffer"];						//		its3DPolyhedronPlacementBuffers[anInflightBufferIndex]
																							//	is nil
	}
	else	//	game is 2D
	{
		//	Release any previously allocated 3D buffers.
		its3DUniformBuffers[anInflightBufferIndex]							= nil;
		its3DPlainCoveringTransformationBuffers[anInflightBufferIndex]		= nil;
		its3DReflectingCoveringTransformationBuffers[anInflightBufferIndex]	= nil;
		its3DPolyhedronPlacementBuffers[anInflightBufferIndex]				= nil;
		
		//	Write current data into the 2D buffers.
		//	The write2D…IntoBuffer methods typically
		//	re-use the existing buffers that we pass into them.
		//	Only in the exceptional case that the required buffer size
		//	has changed will the write2D…IntoBuffer method replace
		//	the passed-in buffer with a new one of the correct size.

		its2DUniformBuffers[anInflightBufferIndex]
			= [self write2DUniformsIntoBuffer:its2DUniformBuffers[anInflightBufferIndex] modelData:md];
		[theDictionary setValue:	its2DUniformBuffers[anInflightBufferIndex]
						 forKey:	@"2D uniform buffer"];

		its2DCoveringTransformationBuffers[anInflightBufferIndex]
			= [self write2DCoveringTransformationsIntoBuffer:its2DCoveringTransformationBuffers[anInflightBufferIndex] modelData:md];
		[theDictionary setValue:	its2DCoveringTransformationBuffers[anInflightBufferIndex]
						 forKey:	@"2D covering transformation buffer"];

		its2DSpritePlacementBuffers[anInflightBufferIndex]
			= [self write2DSpritePlacementsIntoBuffer:its2DSpritePlacementBuffers[anInflightBufferIndex] modelData:md];
		[theDictionary setValue:	its2DSpritePlacementBuffers[anInflightBufferIndex]	//	Clears the key if
						 forKey:	@"2D sprite placement buffer"];						//		its2DSpritePlacementBuffers[anInflightBufferIndex]
																						//	is nil
	}
	
	return theDictionary;
}

- (NSDictionary<NSString *, id> *)prepareInflightDataBuffersForOffscreenRenderingAtSize:(CGSize)anImageSize modelData:(ModelData *)md
{
	NSMutableDictionary<NSString *, id<MTLBuffer>>	*theDictionary;
	id<MTLBuffer>									theOneShot3DUniformBuffer;
	TransformationBufferPair						*theOneShot3DCoveringTransformationBufferPair;
	id<MTLBuffer>									theOneShot3DPolyhedronPlacementBuffer,
													theOneShot2DUniformBuffer,
													theOneShot2DCoveringTransformationBuffer,
													theOneShot2DSpritePlacementBuffer;
	
	UNUSED_PARAMETER(anImageSize);
	
	theDictionary = [[NSMutableDictionary<NSString *, id<MTLBuffer>> alloc] initWithCapacity:16];

	if (GameIs3D(md->itsGame))
	{
		theOneShot3DUniformBuffer = [self write3DUniformsIntoBuffer:nil modelData:md];
		[theDictionary setValue:	theOneShot3DUniformBuffer
						 forKey:	@"3D uniform buffer"];

		theOneShot3DCoveringTransformationBufferPair = [self write3DCoveringTransformationsIntoBuffersPlain:nil reflecting:nil modelData:md];
		[theDictionary setValue:	theOneShot3DCoveringTransformationBufferPair->itsPlainTransformations
						 forKey:	@"3D plain covering transformation buffer"		];
		[theDictionary setValue:	theOneShot3DCoveringTransformationBufferPair->itsReflectingTransformations
						 forKey:	@"3D reflecting covering transformation buffer"	];

		theOneShot3DPolyhedronPlacementBuffer = [self write3DPolyhedronPlacementsIntoBuffer:nil modelData:md];
		[theDictionary setValue:	theOneShot3DPolyhedronPlacementBuffer
						 forKey:	@"3D polyhedron placement buffer"];
	}
	else	//	game is 2D
	{
		theOneShot2DUniformBuffer = [self write2DUniformsIntoBuffer:nil modelData:md];
		[theDictionary setValue:	theOneShot2DUniformBuffer
						 forKey:	@"2D uniform buffer"];

		theOneShot2DCoveringTransformationBuffer = [self write2DCoveringTransformationsIntoBuffer:nil modelData:md];
		[theDictionary setValue:	theOneShot2DCoveringTransformationBuffer
						 forKey:	@"2D covering transformation buffer"];

		theOneShot2DSpritePlacementBuffer = [self write2DSpritePlacementsIntoBuffer:nil modelData:md];
		[theDictionary setValue:	theOneShot2DSpritePlacementBuffer
						 forKey:	@"2D sprite placement buffer"];
	}
	
	return theDictionary;
}

- (bool)wantsClearWithModelData:(ModelData *)md
{
	UNUSED_PARAMETER(md);

	return true;
}

- (ColorP3Linear)clearColorWithModelData:(ModelData *)md
{
#if (SHAPE_OF_SPACE_TICTACTOE == 2)
	return (ColorP3Linear){0.0, 0.0, 0.0, 0.0};	//	transparent
#endif
	
	if (GameIs3D(md->itsGame))
	{
		//	If we ever enable Save Image or Copy Image in Torus Games,
		//	replace this opaque black color (r,g,b,α) = (0,0,0,1)
		//	with a purely transparent color (r,g,b,α) = (0,0,0,0).
		//	In practice the transparent color also shows up
		//	as black on the screen, although I'm not sure whether
		//	that's guaranteed behavior or just an undocumented
		//	implementation detail.
		//
		return (ColorP3Linear){0.0, 0.0, 0.0, 1.0};	//	black
	}
	else	//	game is 2D
	{
		//	For 2D games, let's set the Metal clear color to match the matte's color.
		//	This produces a good effect during the transitions between
		//	the Basic and Repeating view modes -- it gives the impression
		//	that the game board is shrinking or growing, with nothing behind it.
		//
		return (ColorP3Linear) {
					GammaDecode(gMatteColorGammaP3[0]),
					GammaDecode(gMatteColorGammaP3[1]),
					GammaDecode(gMatteColorGammaP3[2]),
					1.0};
	}
}

- (void)encodeCommandsToCommandBuffer:(id<MTLCommandBuffer>)aCommandBuffer
	withRenderPassDescriptor:(MTLRenderPassDescriptor *)aRenderPassDescriptor
	inflightDataBuffers:(NSDictionary<NSString *, id> *)someInflightDataBuffers
	modelData:(ModelData *)md
{
	id<MTLRenderCommandEncoder>	theRenderEncoder;
	
	theRenderEncoder = [aCommandBuffer renderCommandEncoderWithDescriptor:aRenderPassDescriptor];
	if (GameIs3D(md->itsGame))
	{
		[self encode3DCommandsWithEncoder:	theRenderEncoder
					  inflightDataBuffers:	someInflightDataBuffers
						  		modelData:	md];
	}
	else	//	game is 2D
	{
		[self encode2DCommandsWithEncoder:	theRenderEncoder
					  inflightDataBuffers:	someInflightDataBuffers
						  		modelData:	md];
	}
	[theRenderEncoder endEncoding];
}


#pragma mark -
#pragma mark 2D buffers

- (id<MTLBuffer>)write2DUniformsIntoBuffer:(id<MTLBuffer>)aUniformsBuffer modelData:(ModelData *)md
{
	unsigned int			theRequiredBufferLengthInBytes;
	id<MTLBuffer>			theUniformsBuffer;
	Offset2D				theOffset;
	double					the2DViewMagFactor;
	unsigned int			theBackgroundTexReps;
	float					theKleinAxisColors[2][4];	//	premultiplied alpha
	double					theScaleFactor;
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	Placement2D				theScaledHandCursorPlacement;
#endif
	TorusGames2DUniformData	*the2DUniformData;
	unsigned int			thePuzzleSize;
	double					theCellWidthPx;
	unsigned int			i;
	double					theJigsawTexturePlacements[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE][4];	//	( u_min, v_min, u_max - u_min, v_max - v_min ) in bottom-up coordinates

	GEOMETRY_GAMES_ASSERT(
		! GameIs3D(md->itsGame),
		"Internal error:  got 3D game where 2D game was expected");

	theRequiredBufferLengthInBytes = sizeof(TorusGames2DUniformData);
	
	if (aUniformsBuffer != nil && [aUniformsBuffer length] == theRequiredBufferLengthInBytes)
		theUniformsBuffer = aUniformsBuffer;
	else
		theUniformsBuffer = [itsDevice newBufferWithLength:theRequiredBufferLengthInBytes options:MTLResourceStorageModeShared];
	
	theOffset				= md->itsOffset;
	the2DViewMagFactor		= md->its2DViewMagFactor;
	theBackgroundTexReps	= GetNum2DBackgroundTextureRepetitions(md);
	Get2DKleinAxisColors(md, theKleinAxisColors);

	the2DUniformData = (TorusGames2DUniformData *) [theUniformsBuffer contents];
	
	//	itsDragPlacement
	
	//		Keep in mind that the columns in Metal's right-to-left matrix convention
	//		correspond to the rows in our left-to-right convention.

		//	Start with an identity matrix.
	the2DUniformData->itsWorldData.itsDragPlacement = matrix_identity_float3x3;
	
		//	Reflect side-to-side if necessary.
	if (theOffset.itsFlip)
		the2DUniformData->itsWorldData.itsDragPlacement.columns[0][0] = -1.0;
	
		//	Translate.
	the2DUniformData->itsWorldData.itsDragPlacement.columns[2][0] = theOffset.itsH;
	the2DUniformData->itsWorldData.itsDragPlacement.columns[2][1] = theOffset.itsV;
	
		//	If the2DViewMagFactor were 1.0, then we'd want to scale
		//	from the fundamental square's [-0.5, +0.5] coordinates
		//	to Metal's [-1.0, +1.0] coordinates.
		//
		//	If the2DViewMagFactor isn't 1.0 (for example in ViewRepeating it's 1/3)
		//	then we must factor it in too.
		//
	theScaleFactor = 2.0 * the2DViewMagFactor;
	
		//	Rescale by theScaleFactor.
	the2DUniformData->itsWorldData.itsDragPlacement.columns[0][0] *= theScaleFactor;
	the2DUniformData->itsWorldData.itsDragPlacement.columns[1][1] *= theScaleFactor;
	the2DUniformData->itsWorldData.itsDragPlacement.columns[2][0] *= theScaleFactor;
	the2DUniformData->itsWorldData.itsDragPlacement.columns[2][1] *= theScaleFactor;
	
	
	//	Clipping distance
	the2DUniformData->itsWorldData.itsClippingDistance
		= (md->itsViewType == ViewBasicSmall ? the2DViewMagFactor : 1.0);
	
	//	Background texture repetitions
	the2DUniformData->itsBackgroundTexReps	= (float) theBackgroundTexReps;

	//	Klein bottle glide-reflection axes
	the2DUniformData->itsKleinAxisPlacementA	= ConvertPlacementToSIMD(
													& (Placement2D) {0.0, 0.0, false, 0.5*PI, 1.0, 0.01});
	the2DUniformData->itsKleinAxisPlacementB	= ConvertPlacementToSIMD(
													& (Placement2D) {0.5, 0.0, false, 0.5*PI, 1.0, 0.01});
	the2DUniformData->itsKleinAxisColorA		= ConvertVector4ftoSIMDHalf(theKleinAxisColors[0]);
	the2DUniformData->itsKleinAxisColorB		= ConvertVector4ftoSIMDHalf(theKleinAxisColors[1]);
	the2DUniformData->itsKleinAxisTexReps		= (float) NUM_KLEIN_AXIS_TEXTURE_REPETITIONS;

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	//	Hand cursor placement
	theScaledHandCursorPlacement				= GetScaledHandCursorPlacement(md);
	the2DUniformData->itsHandCursorPlacement	= ConvertPlacementToSIMD(&theScaledHandCursorPlacement);
#endif

	//	Game-specific colors etc.
	switch (md->itsGame)
	{
		case Game2DGomoku:
			the2DUniformData->itsGomokuWinLineColor		= (faux_simd_half4) PREMULTIPLY_RGBA( -0.225,  1.042,  0.196,  1.000 );	//	= P3(0, 1, 1/4)
			break;
	
		case Game2DMaze:
			the2DUniformData->itsMazeWallColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  0.000,  0.000,  0.500,  1.000 );
			break;

		case Game2DCrossword:

			thePuzzleSize	= md->itsGameOf.Crossword2D.itsPuzzleSize;
#warning Would we have a problem with itsOnscreenNativeSizePx if we tried exporting at some size other than the default? \
  Yes.  The renderer in the SwiftUI version of this app avoids storing information about any particular target, \
  and thus avoids this issue.  I'm not going to bother to try to fix it in this legacy version.
			theCellWidthPx	= (double) itsOnscreenNativeSizePx.width / (double) thePuzzleSize;

			the2DUniformData->itsCrosswordGridTexReps = thePuzzleSize;

			//	Let the transition zone between an opaque black grid line
			//	and its transparent complement be about 1 pixel wide,
			//	so all the grid lines will appear to have the same width
			//	while still being as sharp as possible.
			//
			//	For computational efficiency we pass the inverse of the transition zone width,
			//	so the fragment function won't have to do the same (relatively expensive!)
			//	floating point division over and over, for each pixel that it processes.
			//
			the2DUniformData->itsCrosswordGridLineTransitionZoneWidthInverse = (float) theCellWidthPx;

			the2DUniformData->itsCrosswordFlashColor			= (faux_simd_half4) PREMULTIPLY_RGBA( -0.22,  1.04,  0.47,  1.00 );	//	= P3(0, 1, 1/2)
			the2DUniformData->itsCrosswordGridColor				= (faux_simd_half4) PREMULTIPLY_RGBA(  0.00,  0.00,  0.00,  1.00 );
			the2DUniformData->itsCrosswordHotCellColor			= (faux_simd_half4) PREMULTIPLY_RGBA( -0.11,  0.52,  1.06,  1.00 );	//	= P3(0, 1/2, 1)
			the2DUniformData->itsCrosswordHotWordColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  0.50,  0.75,  1.00,  1.00 );
			the2DUniformData->itsCrosswordCharacterColor		= (faux_simd_half4) PREMULTIPLY_RGBA(  0.00,  0.00,  0.00,  1.00 );
			the2DUniformData->itsCrosswordPendingCharacterColor	= (faux_simd_half4) PREMULTIPLY_RGBA(  0.00,  0.00,  0.00,  0.50 );	//	50% opaque black
			the2DUniformData->itsCrosswordArrowColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  1.00,  1.00, -0.10,  1.00 );	//	= P3(1, 1, 0)

			break;

		case Game2DWordSearch:

			the2DUniformData->itsWordSearchCharacterColor = (faux_simd_half4) PREMULTIPLY_RGBA(0.00, 0.00, 0.00, 1.00);

			for (i = 0; i < md->itsGameOf.WordSearch2D.itsNumLinesFound; i++)
			{
				the2DUniformData->itsWordSearchMarkedWordColors[i]	//	premultiplied alpha
					= ConvertVector4ftoSIMDHalf(md->itsGameOf.WordSearch2D.itsLines[i].itsColor);
			}
			
			if (md->itsGameOf.WordSearch2D.itsWordSelectionIsPending)
			{
				the2DUniformData->itsWordSearchPendingWordColor	//	premultiplied alpha
					= ConvertVector4ftoSIMDHalf(md->itsGameOf.WordSearch2D.itsColor);
			}
			else
			{
				the2DUniformData->itsWordSearchPendingWordColor = (faux_simd_half4) PREMULTIPLY_RGBA(0.00, 0.00, 0.00, 1.00);	//	unused
			}
			
			the2DUniformData->itsWordSearchHotPointColor = (faux_simd_half4) PREMULTIPLY_RGBA(-0.11, 0.52, 1.06, 1.00);	//	= P3(0, 1/2, 1)

			break;
		
		case Game2DJigsaw:
			Get2DJigsawSpritePlacementsOrTexturePlacements(	md,
															md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize,
															NULL,
															theJigsawTexturePlacements);
			for (i = 0; i < md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i++)
				the2DUniformData->itsJigsawTexturePlacement[i] = ConvertVector4toSIMD(theJigsawTexturePlacements[i]);
			break;
		
		case Game2DChess:
			the2DUniformData->itsChessWhoseTurnHaloColor	= (faux_simd_half4) PREMULTIPLY_RGBA( -0.225,  1.042,  0.471,  1.000 );	//	= P3(0, 1, 1/2)
			the2DUniformData->itsChessGhostColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  1.000,  1.000,  1.000,  0.375 );	//	3/8 opaque
			the2DUniformData->itsChessDragPieceHaloColor	= (faux_simd_half4) PREMULTIPLY_RGBA( -0.112,  0.521,  1.059,  1.000 );	//	= P3(0, 1/2, 1)
			the2DUniformData->itsChessDialBaseColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  0.625,  0.875,  0.938,  1.000 );
			the2DUniformData->itsChessDialSweepColor		= (faux_simd_half4) PREMULTIPLY_RGBA(  1.225, -0.042, -0.020,  1.000 );	//	= P3(1, 0, 0)
			the2DUniformData->itsChessDialRimColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  0.000,  0.000,  0.000,  1.000 );
			the2DUniformData->itsChessDialArrowColor		= (faux_simd_half4) PREMULTIPLY_RGBA(  0.000,  0.000,  0.000,  1.000 );
			break;

		case Game2DPool:
			the2DUniformData->itsPoolHotSpotColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  1.000,  1.000, -0.098,  1.000 );	//	= P3(1, 1, 0)
			break;

		case Game2DApples:
			the2DUniformData->itsApplesNumeralColor			= (faux_simd_half4) PREMULTIPLY_RGBA(  0.000,  0.000,  1.098,  1.000 );	//	= P3(0, 0, 1)
			break;

		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game3DTicTacToe:
		case Game3DMaze:
		case NumGameTypes:
			break;
	}

	return theUniformsBuffer;
}

- (id<MTLBuffer>)write2DCoveringTransformationsIntoBuffer:(id<MTLBuffer>)aCoveringTransformationBuffer modelData:(ModelData *)md
{
	signed int							n;	//	signed integer for comparison with dh and dv
	unsigned int						theNumCoveringTransformations,
										theRequiredBufferLengthInBytes;
	id<MTLBuffer>						theCoveringTransformationBuffer;
	TorusGames2DCoveringTransformation	*theCoveringTransformations,
										*theCoveringTransformation;
	signed int							dh,
										dv;

	//	Render a 3×3 set of translates to cover a 1×1 domain,
	//	    or a 5×5 set of translates to cover a 3×3 domain.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
		case ViewBasicSmall:
			n = 1;
			break;
		
		case ViewRepeating:
			n = 2;
			break;
		
		default:
			n = 0;	//	should never occur
			break;
	}
	theNumCoveringTransformations = (2*n + 1) * (2*n + 1);
	
	//	Make sure aCoveringTransformationBuffer has the correct size.
	//	If it doesn't, replace it.
	theRequiredBufferLengthInBytes = theNumCoveringTransformations * sizeof(TorusGames2DCoveringTransformation);
	if (aCoveringTransformationBuffer != nil && [aCoveringTransformationBuffer length] == theRequiredBufferLengthInBytes)
		theCoveringTransformationBuffer = aCoveringTransformationBuffer;
	else
		theCoveringTransformationBuffer = [itsDevice newBufferWithLength:theRequiredBufferLengthInBytes options:MTLResourceStorageModeShared];
	
	//	Populate theCoveringTransformationBuffer.
	theCoveringTransformations	= (TorusGames2DCoveringTransformation *) [theCoveringTransformationBuffer contents];
	theCoveringTransformation	= theCoveringTransformations;
	for (dh = -n; dh <= +n; dh++)
	{
		for (dv = -n; dv <= +n; dv++)
		{
			*theCoveringTransformation = matrix_identity_float3x3;
			
			if (md->itsTopology == Topology2DKlein
			 && (((unsigned int)dv) % 2 == 1))	//	Caution: when dv is negative and odd,
			 									//	dv % 2 == -1 (not +1 as I had been expecting)
			{
				theCoveringTransformation->columns[0][0] = -1.0;
			}
			
			theCoveringTransformation->columns[2][0] = dh;
			theCoveringTransformation->columns[2][1] = dv;
			
			theCoveringTransformation++;
		}
	}

	//	Return the buffer, which may occasionally
	//	be a different buffer from the one the caller passed in.
	return theCoveringTransformationBuffer;
}

- (id<MTLBuffer>)write2DSpritePlacementsIntoBuffer:(id<MTLBuffer>)aSpritePlacementBuffer modelData:(ModelData *)md
{
	unsigned int						theNumSprites;
	Placement2D							*theSpritePlacements			= NULL;
	unsigned int						theRequiredBufferLengthInBytes;
	id<MTLBuffer>						theSpritePlacementBuffer		= nil;
	TorusGames2DSpritePlacementMatrix	*theSpriteMatrices				= NULL;
	unsigned int						i;

	theNumSprites = GetNum2DSprites(md);
	
	if (theNumSprites == 0)	//	may in principle occur, if no sprites are present
		return nil;

	theSpritePlacements = (Placement2D *) GET_MEMORY(theNumSprites * sizeof(Placement2D));
	GEOMETRY_GAMES_ASSERT(theSpritePlacements != NULL, "out of memory");
	
	Get2DSpritePlacements(md, theNumSprites, theSpritePlacements);

	//	Make sure aSpritePlacementBuffer has the correct size.
	//	If it doesn't, replace it.
	theRequiredBufferLengthInBytes = theNumSprites * sizeof(TorusGames2DSpritePlacementMatrix);
	if (aSpritePlacementBuffer != nil && [aSpritePlacementBuffer length] == theRequiredBufferLengthInBytes)
		theSpritePlacementBuffer = aSpritePlacementBuffer;
	else
		theSpritePlacementBuffer = [itsDevice newBufferWithLength:theRequiredBufferLengthInBytes options:MTLResourceStorageModeShared];
	
	//	Populate theSpritePlacementBuffer.
	theSpriteMatrices = (TorusGames2DSpritePlacementMatrix *) [theSpritePlacementBuffer contents];
	for (i = 0; i < theNumSprites; i++)
		theSpriteMatrices[i] = ConvertPlacementToSIMD(&theSpritePlacements[i]);

	FREE_MEMORY_SAFELY(theSpritePlacements);

	//	Return the buffer, which may occasionally
	//	be a different buffer from the one the caller passed in.
	return theSpritePlacementBuffer;
}


#pragma mark -
#pragma mark 3D buffers

- (id<MTLBuffer>)write3DUniformsIntoBuffer:(id<MTLBuffer>)aUniformsBuffer modelData:(ModelData *)md
{
	unsigned int			theRequiredBufferLengthInBytes;
	id<MTLBuffer>			theUniformsBuffer;
	TorusGames3DUniformData	*the3DUniformData;
	double					theProjectionMatrix[4][4],
							theFrameCellIntoWorld[4][4],
							theTilingIntoWorld[4][4];
	unsigned int			i,
							j;
	double					theTexCoordShift[2],
							theBrightnessFactor;

	GEOMETRY_GAMES_ASSERT(
		GameIs3D(md->itsGame),
		"Internal error:  got 2D game where 3D game was expected");

	theRequiredBufferLengthInBytes = sizeof(TorusGames3DUniformData);
	
	if (aUniformsBuffer != nil && [aUniformsBuffer length] == theRequiredBufferLengthInBytes)
		theUniformsBuffer = aUniformsBuffer;
	else
		theUniformsBuffer = [itsDevice newBufferWithLength:theRequiredBufferLengthInBytes options:MTLResourceStorageModeShared];

	the3DUniformData = (TorusGames3DUniformData *) [theUniformsBuffer contents];


	//	for the scene as a whole

	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);
	Matrix44Product(md->its3DTilingIntoFrameCell, theFrameCellIntoWorld, theTilingIntoWorld);

	the3DUniformData->itsWorldData.itsTilingIntoFrameCell	= ConvertMatrix44ToSIMD(md->its3DTilingIntoFrameCell);
	the3DUniformData->itsWorldData.itsFrameCellIntoWorld	= ConvertMatrix44ToSIMD(theFrameCellIntoWorld);
	the3DUniformData->itsWorldData.itsTilingIntoWorld		= ConvertMatrix44ToSIMD(theTilingIntoWorld);

	MakeProjectionMatrix3D(md, theProjectionMatrix);
	the3DUniformData->itsWorldData.itsProjection = ConvertMatrix44ToSIMD(theProjectionMatrix);
	
	the3DUniformData->itsWorldData.itsClippingRadiusForRepeatingView = (float) CLIPPING_RADIUS_FOR_REPEATING_VIEW;
	
	Make3DFogDistances(	md,
					//	&the3DUniformData->itsWorldData.itsFogBegin,
						&the3DUniformData->itsWorldData.itsFogEnd,
						&the3DUniformData->itsWorldData.itsFogScale);


	//	for frame cell walls

	the3DUniformData->itsWorldData.itsAperture[0] = md->its3DCurrentAperture;
	the3DUniformData->itsWorldData.itsAperture[1] = 1.0 - md->its3DCurrentAperture;
	
	the3DUniformData->itsPatternParityPlus	= +1.0;
	the3DUniformData->itsPatternParityMinus	= -1.0;
	
	for (i = 0; i < NUM_FRAME_WALLS; i++)
	{
		the3DUniformData->itsTexCoordShifts[i] = ConvertVector2toSIMD(
			Make3DTexCoordShift(md->its3DTilingIntoFrameCell, i, theTexCoordShift));
	}

	for (i = 0; i < NUM_AVAILABLE_WALL_COLORS; i++)
		the3DUniformData->itsWallColors[i] = ConvertVector4ftoSIMDHalf(gWallColors[i]);

	//	Darken the walls during Simulation3DResetGameAsDomain.
	//	Even though the app is merely shrinking the frame cell as it spins,
	//	the user perceives the cube as spinning off to infinity,
	//	so we should darken it.
	//
	if (md->itsSimulationStatus == Simulation3DResetGameAsDomainPart1
	 || md->itsSimulationStatus == Simulation3DResetGameAsDomainPart2
	 || md->itsSimulationStatus == Simulation3DResetGameAsDomainPart3)
	{
		theBrightnessFactor = md->its3DResetGameScaleFactor
							* (2.0 - md->its3DResetGameScaleFactor);

		for (i = 0; i < NUM_AVAILABLE_WALL_COLORS; i++)
			for (j = 0; j < 3; j++)
				the3DUniformData->itsWallColors[i][j] *= theBrightnessFactor;
	}
	

	//	for game content

	the3DUniformData->itsTicTacToeNeutralColor	= (faux_simd_half4) PREMULTIPLY_RGBA( 1.000,  1.000,  1.000,  1.000 );
	the3DUniformData->itsTicTacToeXColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 1.225, -0.042, -0.020,  1.000 );	//	= P3(1,0,0)
	the3DUniformData->itsTicTacToeOColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 0.000,  0.000,  1.098,  1.000 );	//	= P3(0,0,1)
	the3DUniformData->itsTicTacToeWinLineColor	= (faux_simd_half4) PREMULTIPLY_RGBA( 1.000,  1.000,  1.000,  1.000 );

	the3DUniformData->itsMazeTracksColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 1.000,  1.000,  1.000,  1.000 );
	the3DUniformData->itsMazeSliderColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 0.000,  0.000,  1.098,  1.000 );	//	= P3(0,0,1)
	if (md->itsGameIsOver)
	{
		the3DUniformData->itsMazeGoalOuterColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 0.000,  1.000,  0.000,  1.000 );
		the3DUniformData->itsMazeGoalInnerColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 0.000,  0.500,  0.000,  1.000 );
	}
	else
	{
		the3DUniformData->itsMazeGoalOuterColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 1.225, -0.042, -0.020,  1.000 );	//	= P3(1,0,0)
		the3DUniformData->itsMazeGoalInnerColor		= (faux_simd_half4) PREMULTIPLY_RGBA( 0.250,  0.250,  0.250,  1.000 );
	}
	
	return theUniformsBuffer;
}

- (TransformationBufferPair *)write3DCoveringTransformationsIntoBuffersPlain:(id<MTLBuffer>)aPlainCoveringTransformationBuffer
									reflecting:(id<MTLBuffer>)aReflectingCoveringTransformationBuffer modelData:(ModelData *)md
{
	double								theFrameCellIntoWorld[4][4],
										theTilingIntoWorld[4][4];
	unsigned int						theMaxNumPlainCoveringTransformations		= 0,	//	initialize to suppress compiler warnings
										theMaxNumReflectingCoveringTransformations	= 0,	//	initialize to suppress compiler warnings
										theRequiredPlainBufferLengthInBytes,
										theRequiredReflectingBufferLengthInBytes;
	id<MTLBuffer>						thePlainCoveringTransformationBuffer,
										theReflectingCoveringTransformationBuffer;
	double								theBoundingBoxCornersInGameCell[2][4],	//	a pair of opposite corners of the bounding box,
																				//		in [-0.5, +0.5] game cell coordinates
										theBoundingBoxCornersInTiling[2][4],	//	same thing, but in tiling coordinates
										theBoundingBoxCorners[2][4];			//	same thing, but in frame cell coordinates (ViewBasicLarge)
																				//						 or world coordinates (ViewRepeating)
	TorusGames3DCoveringTransformation	*thePlainCoveringTransformations,
										*theReflectingCoveringTransformations,
										*thePlainCoveringTransformation,
										*theReflectingCoveringTransformation;
	signed int							x,	//	Integer coordinates of tile's center.
										y,
										z,
										theTilingSize;
	double								theClippingRadiusSquared;
	unsigned int						i,
										j;
	double								theGameCellIntoTiling[4][4];
	simd_float4x4						theGameCellIntoTilingAsSIMD;
	double								theSwapValue;
	bool								theTranslatedContentMayBeVisible;
	double								theFrustumHalfWidth,
										theCameraToBoundingBoxDistanceSquared;
	TransformationBufferPair			*theTransformationBufferPair;
	double								theFillerTransformationAsCMatrix[4][4];
	simd_float4x4						theFillerTransformation;
	

	//	Convert its3DFrameCellIntoWorld from a spin vector in Spin(3)
	//	to a 4×4 matrix in SO(3), padded out with zero translational part.
	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);
	
	//	Pre-compute theTilingIntoWorld.
	Matrix44Product(md->its3DTilingIntoFrameCell, theFrameCellIntoWorld, theTilingIntoWorld);

	//	Precompute the maximum number of plain and reflecting transformations that we might ever need.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	The hard-coded products below allow for all potentially visible content
			//	in a 3×3×3 set of translates.
			switch (md->itsTopology)
			{
				case Topology3DTorus:
				case Topology3DQuarterTurn:
				case Topology3DHalfTurn:
					theMaxNumPlainCoveringTransformations		= 3 * 3 * 3;
					theMaxNumReflectingCoveringTransformations	= 0;
					break;
				
				case Topology3DKlein:
					theMaxNumPlainCoveringTransformations		= 1 * 3 * 3;	//	y = 0  layer is plain
					theMaxNumReflectingCoveringTransformations	= 2 * 3 * 3;	//	y = ±1 layers are reflecting
					break;
				
				default:
					GEOMETRY_GAMES_ABORT("invalid 3D topology");
					break;
			}
			break;
		
		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("The 3D games don't use ViewBasicSmall.");
			break;
		
		case ViewRepeating:
			//	The following estimates are several times larger than they need to be.
			//	But the total amount of memory is nevertheless quite modest.
			//	For example, when TILING_SIZE_FOR_REPEATING_VIEW = 3, we're allowing
			//	for a 7×7×7 cube of covering transformations, each represented by a 4×4 matrix
			//	of 4-byte floats, for a total buffer size of (7*7*7)*(4*4)*4 = 21952 bytes.
			//	So for simplicity let's go ahead and allocate the whole thing,
			//	even though we'll use only about 1/6 of it in the orientable case,
			//	and about 1/12 of it in each buffer in the non-orientable case.
			switch (md->itsTopology)
			{
				case Topology3DTorus:
				case Topology3DQuarterTurn:
				case Topology3DHalfTurn:
					theMaxNumPlainCoveringTransformations		= TOTAL_CUBE_CELLS_FOR_REPEATING_VIEW;
					theMaxNumReflectingCoveringTransformations	= 0;
					break;
				
				case Topology3DKlein:
					theMaxNumPlainCoveringTransformations		= TOTAL_CUBE_CELLS_FOR_REPEATING_VIEW;	//	even y layers are plain
					theMaxNumReflectingCoveringTransformations	= TOTAL_CUBE_CELLS_FOR_REPEATING_VIEW;	//	odd  y layers are reflecting
					break;
				
				default:
					GEOMETRY_GAMES_ABORT("invalid 3D topology");
					break;
			}
			break;
	}
	
	//	Make sure each buffer has the correct size.
	//	If it doesn't, replace it.

	theRequiredPlainBufferLengthInBytes = theMaxNumPlainCoveringTransformations * sizeof(TorusGames3DCoveringTransformation);
	if (aPlainCoveringTransformationBuffer != nil
	 && [aPlainCoveringTransformationBuffer length] == theRequiredPlainBufferLengthInBytes)
	{
		thePlainCoveringTransformationBuffer = aPlainCoveringTransformationBuffer;
	}
	else
	{
		//	Note:  -newBufferWithLength:options: will not allocate a zero-byte buffer,
		//	so for future robustness let's explicitly set the reference to nil
		//	in such cases, without calling -newBufferWithLength:options: .
		
		if (theRequiredPlainBufferLengthInBytes > 0)	//	always true
			thePlainCoveringTransformationBuffer = [itsDevice newBufferWithLength:theRequiredPlainBufferLengthInBytes options:MTLResourceStorageModeShared];
		else
			thePlainCoveringTransformationBuffer = nil;
	}

	theRequiredReflectingBufferLengthInBytes = theMaxNumReflectingCoveringTransformations * sizeof(TorusGames3DCoveringTransformation);
	if (aReflectingCoveringTransformationBuffer != nil
	 && [aReflectingCoveringTransformationBuffer length] == theRequiredReflectingBufferLengthInBytes)
	{
		theReflectingCoveringTransformationBuffer = aReflectingCoveringTransformationBuffer;
	}
	else
	{
		//	See the Note immediately above, regarding zero-byte buffers.
		
		if (theRequiredReflectingBufferLengthInBytes > 0)	//	true only for Klein space
			theReflectingCoveringTransformationBuffer = [itsDevice newBufferWithLength:theRequiredReflectingBufferLengthInBytes options:MTLResourceStorageModeShared];
		else
			theReflectingCoveringTransformationBuffer = nil;
	}

	//	Let the game specify an axis-aligned bounding box
	//	that encloses all its content, including any content
	//	that might extend beyond the game cell proper.
	//
	Get3DAxisAlignedBoundingBox(md, theBoundingBoxCornersInGameCell);
	
	//	Populate the buffers.

	thePlainCoveringTransformations
		= (TorusGames3DCoveringTransformation *) [thePlainCoveringTransformationBuffer      contents];
	theReflectingCoveringTransformations
		= (TorusGames3DCoveringTransformation *) [theReflectingCoveringTransformationBuffer contents];

	thePlainCoveringTransformation		= thePlainCoveringTransformations;
	theReflectingCoveringTransformation	= theReflectingCoveringTransformations;

	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	Consider a 3×3×3 set of translates, and decide
			//	which ones' content may intersect the frame cell.
			theTilingSize = +1;
			break;
		
		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("The 3D games don't use ViewBasicSmall.");
			break;
		
		case ViewRepeating:
			//	Consider a larger set of translates, and decide
			//	which ones' content may intersect the view frustum.
			theTilingSize = TILING_SIZE_FOR_REPEATING_VIEW;
			break;
	}
	
	if (md->itsViewType == ViewRepeating)
	{
		//		Note:  Rather than blindly assuming that game content may extend
		//		up to 0.5 units beyond the game cell, we could instead use
		//		the actual value of theBoundingBoxCornersInGameCell to get
		//		a larger value for the clipping radius.  But this larger value
		//		would make things more complicated in the GPU vertex function,
		//		which must compute clipping and fog values consistent
		//		with the clipping radius that we choose here.  So better
		//		to keep things simple and consistent.  If we ever feel we need
		//		a deeper tiling, we can increase the TILING_SIZE_FOR_REPEATING_VIEW.
		//
		theClippingRadiusSquared = CLIPPING_RADIUS_FOR_REPEATING_VIEW
								 * CLIPPING_RADIUS_FOR_REPEATING_VIEW;
	}
	else
	{
		theClippingRadiusSquared = 0.0;	//	unused, but initialized to suppress compiler warning
	}

	for (x = -theTilingSize; x <= +theTilingSize; x++)
	{
		for (y = -theTilingSize; y <= +theTilingSize; y++)
		{
			for (z = -theTilingSize; z <= theTilingSize; z++)
			{
				//	Compute the element of the covering transformation group
				//	that takes the origin to the point (x,y,z).
				Make3DGameCellIntoTiling(theGameCellIntoTiling, x, y, z, md->itsTopology);
				theGameCellIntoTilingAsSIMD = ConvertMatrix44ToSIMD(theGameCellIntoTiling);

				//	Transfer the bounding box corners...
				for (i = 0; i < 2; i++)
				{
					//	from game cell coordinates to tiling coordinates...
					Matrix44RowVectorTimesMatrix(
						theBoundingBoxCornersInGameCell[i],
						theGameCellIntoTiling,
						theBoundingBoxCornersInTiling[i]);

					//	and thence to either frame cell coordinates or world coordinates,
					//	as appropriate for visibility testing.
					switch (md->itsViewType)
					{
						case ViewBasicLarge:
							Matrix44RowVectorTimesMatrix(
								theBoundingBoxCornersInTiling[i],
								md->its3DTilingIntoFrameCell,
								theBoundingBoxCorners[i]);	//	in frame cell coordinates
							break;
						
						case ViewBasicSmall:
							GEOMETRY_GAMES_ABORT("The 3D games don't use ViewBasicSmall.");
							break;
						
						case ViewRepeating:
							Matrix44RowVectorTimesMatrix(
								theBoundingBoxCornersInTiling[i],
								theTilingIntoWorld,
								theBoundingBoxCorners[i]);	//	in world coordinates
							break;
					}
				}
			
				//	Swap entries between the two corner vectors so that
				//	the smaller value of each coordinate ends up in theBoundingBoxCorners[0] and
				//	the larger  value of each coordinate ends up in theBoundingBoxCorners[1]
				for (j = 0; j < 3; j++)
				{
					if (theBoundingBoxCorners[0][j] > theBoundingBoxCorners[1][j])
					{
						theSwapValue				= theBoundingBoxCorners[0][j];
						theBoundingBoxCorners[0][j]	= theBoundingBoxCorners[1][j];
						theBoundingBoxCorners[1][j]	= theSwapValue;
					}
				}

				//	Is the bounding box at least partially visible?
				switch (md->itsViewType)
				{
					case ViewBasicLarge:
						//	Does the bounding box intersect the frame cell?
						//
						//		Lemma:  The bounding box doesn't intersect the frame cell
						//		iff it lies wholly to one side or the other in at least one coordinate.
						//
						theTranslatedContentMayBeVisible = true;
						for (j = 0; j < 3; j++)
						{
							if (theBoundingBoxCorners[1][j] <= -0.5		//	bounding box lies wholly "below" frame cell
							 || theBoundingBoxCorners[0][j] >= +0.5)	//	bounding box lies wholly "above" frame cell
							{
								theTranslatedContentMayBeVisible = false;
							}
						}
						break;
					
					case ViewBasicSmall:
						GEOMETRY_GAMES_ABORT("The 3D games don't use ViewBasicSmall.");
						break;
					
					case ViewRepeating:

						//	Does the bounding box intersect the view frustum?
						if (theBoundingBoxCorners[1][2] <= -0.5)	//	z_max ≤ 0.5 ?
						{
							//	The bounding box lies wholly behind the near clipping plane.
							//	("Behind" meaning more negative, so it gets clipped away and isn't visible.)
							theTranslatedContentMayBeVisible = false;
						}
						else
						{
							//	Unlike in ViewBasicLarge, where the frame cell may rotate freely,
							//	in ViewRepeating the frame cell must align with the world-space axes.
							//
							//		Lemma:  If the bounding box intersects the view frustum at all,
							//		the bounding box's far face (the face with maximum z value)
							//		must intersect it.
							//
							
							//	In light of that lemma, we may restrict out attention to the 2D plane
							//	containing the bounding box's far face, at z = z_max.
							//	That plane slices through the view frustum in a square of half-width z_max + 1.
							theFrustumHalfWidth = theBoundingBoxCorners[1][2] + 1.0;	//	theFrustumHalfWidth ≥ +0.5
							
							//	To test whether the view frustum's square cross-section intersects
							//	the bounding box's far face, we may apply the same logic used
							//	in the ViewBasicLarge case immediately above, but now for 2D squares
							//	instead of 3D cubes.
							theTranslatedContentMayBeVisible = true;
							for (j = 0; j < 2; j++)	//	consider x and y coordinates only, not z
							{
								if (theBoundingBoxCorners[1][j] <= -theFrustumHalfWidth		//	bounding box lies wholly "below" frustum cross-section
								 || theBoundingBoxCorners[0][j] >= +theFrustumHalfWidth)	//	bounding box lies wholly "above" frustum cross-section
								{
									theTranslatedContentMayBeVisible = false;
								}
							}
						}

						//	Does the bounding box intersect the clipping sphere?
						if (theTranslatedContentMayBeVisible)	//	For efficiency, test the clipping sphere
																//	iff we already found that the bounding box
																//	intersects the view frustum.
						{
							//	Code inspired by
							//
							//		James Arvo. “A Simple Method for Box-Sphere Intersection Testing”.
							//		In Graphics Gems, Academic Press, 1990, pp. 335–339.
							//
							
							theCameraToBoundingBoxDistanceSquared = 0.0;

							if (theBoundingBoxCorners[0][0] >=  0.0)	//	bounding box wholly to the right of the camera?
								theCameraToBoundingBoxDistanceSquared += theBoundingBoxCorners[0][0] * theBoundingBoxCorners[0][0];

							if (theBoundingBoxCorners[1][0] <=  0.0)	//	bounding box wholly to the left of the camera?
								theCameraToBoundingBoxDistanceSquared += theBoundingBoxCorners[1][0] * theBoundingBoxCorners[1][0];

							if (theBoundingBoxCorners[0][1] >=  0.0)	//	bounding box wholly above the camera?
								theCameraToBoundingBoxDistanceSquared += theBoundingBoxCorners[0][1] * theBoundingBoxCorners[0][1];

							if (theBoundingBoxCorners[1][1] <=  0.0)	//	bounding box wholly below the camera?
								theCameraToBoundingBoxDistanceSquared += theBoundingBoxCorners[1][1] * theBoundingBoxCorners[1][1];

							if (theBoundingBoxCorners[0][2] >= -1.0)	//	bounding box wholly in front of the camera? (usually true)
								theCameraToBoundingBoxDistanceSquared += (theBoundingBoxCorners[0][2] - (-1.0)) * (theBoundingBoxCorners[0][2] - (-1.0));

							if (theBoundingBoxCorners[1][2] <= -1.0)	//	bounding box wholly behind the camera?      (never true, I think)
								theCameraToBoundingBoxDistanceSquared += (-1.0 - theBoundingBoxCorners[1][2]) * (-1.0 - theBoundingBoxCorners[1][2]);
							
							
							if (theCameraToBoundingBoxDistanceSquared >= theClippingRadiusSquared)
								theTranslatedContentMayBeVisible = false;
						}

						break;
				}

				if (theTranslatedContentMayBeVisible)
				{
					switch (md->itsTopology)
					{
						case Topology3DTorus:
						case Topology3DQuarterTurn:
						case Topology3DHalfTurn:

							GEOMETRY_GAMES_ASSERT(
								thePlainCoveringTransformation - thePlainCoveringTransformations < theMaxNumPlainCoveringTransformations,
								"The plain covering transformation buffer is too small.");

							*thePlainCoveringTransformation++ = theGameCellIntoTilingAsSIMD;
							
							break;
						
						case Topology3DKlein:
							if (y & 0x00000001)	//	y is odd
							{
								GEOMETRY_GAMES_ASSERT(
									theReflectingCoveringTransformation - theReflectingCoveringTransformations < theMaxNumReflectingCoveringTransformations,
									"The reflecting covering transformation buffer is too small.");

								*theReflectingCoveringTransformation++ = theGameCellIntoTilingAsSIMD;
							}
							else				//	y is even
							{
								GEOMETRY_GAMES_ASSERT(
									thePlainCoveringTransformation - thePlainCoveringTransformations < theMaxNumPlainCoveringTransformations,
									"The plain covering transformation buffer is too small.");

								*thePlainCoveringTransformation++ = theGameCellIntoTilingAsSIMD;
							}
							break;
						
						default:
							GEOMETRY_GAMES_ABORT("invalid 3D topology");
							break;
					}
				}
			}
		}
	}

	
	//	We reallocate the buffers only when the maximum possible needed capacity changes
	//	(for example when changing between ViewBasicLarge and ViewRepeating),
	//	not when the actual needed capacity changes (for example when scrolling).
	//	So typically there'll be unused buffer space.
	//	Fill the unused buffer entries with zero matrices,
	//	so the rendering code will know which buffer entries are in use and which are not.

	Matrix44Zero(theFillerTransformationAsCMatrix);
	theFillerTransformation = ConvertMatrix44ToSIMD(theFillerTransformationAsCMatrix);

	while (thePlainCoveringTransformation - thePlainCoveringTransformations < theMaxNumPlainCoveringTransformations)
		*thePlainCoveringTransformation++ = theFillerTransformation;

	while (theReflectingCoveringTransformation - theReflectingCoveringTransformations < theMaxNumReflectingCoveringTransformations)
		*theReflectingCoveringTransformation++ = theFillerTransformation;

	//	To return the possibly re-created plain and reflecting buffers to the caller,
	//	package them up as a single TransformationBufferPair.
	theTransformationBufferPair = [[TransformationBufferPair alloc] init];
	theTransformationBufferPair->itsPlainTransformations		= thePlainCoveringTransformationBuffer;
	theTransformationBufferPair->itsReflectingTransformations	= theReflectingCoveringTransformationBuffer;
	
	return theTransformationBufferPair;
}

- (id<MTLBuffer>)write3DPolyhedronPlacementsIntoBuffer:(id<MTLBuffer>)aPolyhedronPlacementBuffer modelData:(ModelData *)md
{
	unsigned int								theNumPolyhedra;
	TorusGames3DPolyhedronPlacementAsCArrays	*thePolyhedronPlacementsAsCArrays	= NULL;
	unsigned int								theRequiredBufferLengthInBytes;
	id<MTLBuffer>								thePolyhedronPlacementBuffer		= nil;
	TorusGames3DPolyhedronPlacementAsSIMD		*thePolyhedronPlacementsAsSIMD		= NULL;
	unsigned int								i;

	theNumPolyhedra = GetNum3DPolyhedra(md);
	
	if (theNumPolyhedra == 0)	//	may in principle occur, if no polyhedra are present
		return nil;

	thePolyhedronPlacementsAsCArrays = (TorusGames3DPolyhedronPlacementAsCArrays *)
		GET_MEMORY(theNumPolyhedra * sizeof(TorusGames3DPolyhedronPlacementAsCArrays));
	GEOMETRY_GAMES_ASSERT(thePolyhedronPlacementsAsCArrays != NULL, "out of memory");
	
	Get3DPolyhedronPlacements(md, theNumPolyhedra, thePolyhedronPlacementsAsCArrays);

	//	Make sure aPolyhedronPlacementBuffer has the correct size.
	//	If it doesn't, replace it.
	theRequiredBufferLengthInBytes = theNumPolyhedra * sizeof(TorusGames3DPolyhedronPlacementAsSIMD);
	if (aPolyhedronPlacementBuffer != nil && [aPolyhedronPlacementBuffer length] == theRequiredBufferLengthInBytes)
		thePolyhedronPlacementBuffer = aPolyhedronPlacementBuffer;
	else
		thePolyhedronPlacementBuffer = [itsDevice newBufferWithLength:theRequiredBufferLengthInBytes options:MTLResourceStorageModeShared];
	
	//	Populate thePolyhedronPlacementBuffer.
	thePolyhedronPlacementsAsSIMD = (TorusGames3DPolyhedronPlacementAsSIMD *) [thePolyhedronPlacementBuffer contents];
	for (i = 0; i < theNumPolyhedra; i++)
	{
		thePolyhedronPlacementsAsSIMD[i].itsDilation				= ConvertVector3toSIMD (thePolyhedronPlacementsAsCArrays[i].itsDilation				);
		thePolyhedronPlacementsAsSIMD[i].itsIsometricPlacement		= ConvertMatrix44ToSIMD(thePolyhedronPlacementsAsCArrays[i].itsIsometricPlacement	);
		thePolyhedronPlacementsAsSIMD[i].itsExtraClippingCovector	= ConvertVector4toSIMD (thePolyhedronPlacementsAsCArrays[i].itsExtraClippingCovector);
	}

	FREE_MEMORY_SAFELY(thePolyhedronPlacementsAsCArrays);

	//	Return the buffer, which may occasionally
	//	be a different buffer from the one the caller passed in.
	return thePolyhedronPlacementBuffer;
}


#pragma mark -
#pragma mark 2D masks

- (id<MTLTexture>)makeMazeMaskWithModelData:(ModelData *)md	//	returns a greyscale mask (using MTLPixelFormatR8Unorm)
{
	unsigned short					theLog2PixelsPerRow,
									thePixelsPerRow,
									theCellsPerRow,
									thePixelsPerCell,
									theLineHalfWidth;
	MazeMaskParameters				theMazeMaskParameters;
	unsigned int					theRow,
									theCol;
	MTLTextureDescriptor			*theDescriptor;
	id<MTLTexture>					theMaskTexture	= nil;
	id<MTLCommandBuffer>			theCommandBuffer;
	id<MTLComputeCommandEncoder>	theComputeEncoder;
	NSUInteger						theThreadgroupWidth,
									theThreadgroupHeight,
									theThreadExecutionWidth;

	if (md->itsGameOf.Maze2D.itsMazeIsValid)
	{
		//	The cell size in pixels is inversely proprotional to the number of cells per row.
		//	For the current case of thePixelsPerRow = 1024
		//
		//		maze size (cells per row)		   2    4    8   16   32   64
		//		cell size (pixels per cell)		 512  256  128   64   32   16
		//		total size (pixels per row)		1024 1024 1024 1024 1024 1024
		//
		theLog2PixelsPerRow	= 10;								//	log₂(thePixelsPerRow)
		thePixelsPerRow		= (0x0001 << theLog2PixelsPerRow);	//	= 2¹⁰ = 1024
		theCellsPerRow		= md->itsGameOf.Maze2D.itsSize;		//	=  4,   8,  16 or 32, according to md->itsDifficultyLevel
		thePixelsPerCell	= thePixelsPerRow / theCellsPerRow;	//	= 256, 128, 64 or 32, according to md->itsDifficultyLevel

		//	Within each 16n×16n cell, the outer n-pixel boundary
		//	will accommodate the cell's walls.
		theLineHalfWidth	= thePixelsPerCell / 16;
		
		//	Package up those parameters for the GPU compute function.
		theMazeMaskParameters.itsCellsPerRow		= theCellsPerRow;
		theMazeMaskParameters.itsPixelsPerCell		= thePixelsPerCell;
		theMazeMaskParameters.itsLineHalfWidth		= theLineHalfWidth;
		theMazeMaskParameters.itsKleinBottleFlag	= (md->itsTopology == Topology2DKlein);
		GEOMETRY_GAMES_ASSERT(
			theCellsPerRow <= MAX_MAZE_CELLS_PER_ROW,
			"MAX_MAZE_CELLS_PER_ROW is too small");
		for (theRow = 0; theRow < theCellsPerRow; theRow++)
		{
			for (theCol = 0; theCol < theCellsPerRow; theCol++)
			{
				theMazeMaskParameters.itsWalls[theRow][theCol]
					= (md->itsGameOf.Maze2D.itsCells[theRow][theCol].itsWalls[WallWest]  ? 0x08 : 0x00)
					| (md->itsGameOf.Maze2D.itsCells[theRow][theCol].itsWalls[WallSouth] ? 0x04 : 0x00)
					| (md->itsGameOf.Maze2D.itsCells[theRow][theCol].itsWalls[WallEast]  ? 0x02 : 0x00)
					| (md->itsGameOf.Maze2D.itsCells[theRow][theCol].itsWalls[WallNorth] ? 0x01 : 0x00);
			}
		}
		
		//	Create a greyscale mask (using MTLPixelFormatR8Unorm)
		//	instead of an alpha mask (MTLPixelFormatA8Unorm) so that
		//	the GPU can generate mipmaps for it.  More generally,
		//	MTLPixelFormatR8Unorm is a "shader-writable" pixel format,
		//	while MTLPixelFormatA8Unorm is not.
		//
		theDescriptor = [MTLTextureDescriptor
							texture2DDescriptorWithPixelFormat:	MTLPixelFormatR8Unorm
													width:		thePixelsPerRow
													height:		thePixelsPerRow
													mipmapped:	YES];
		[theDescriptor setUsage:(MTLTextureUsageShaderRead | MTLTextureUsageShaderWrite)];
		[theDescriptor setStorageMode:MTLStorageModePrivate];
		theMaskTexture = [itsDevice newTextureWithDescriptor:theDescriptor];

		//	Let the GPU draw the Maze walls into theMaskTexture.
		
		theCommandBuffer = [itsCommandQueue commandBuffer];
		
		theComputeEncoder = [theCommandBuffer computeCommandEncoder];
		[theComputeEncoder setLabel:@"make Maze mask"];
		[theComputeEncoder setComputePipelineState:itsMakeMazeMaskPipelineState];
		[theComputeEncoder setTexture:theMaskTexture atIndex:TextureIndexCFImage];
		[theComputeEncoder setBytes:&theMazeMaskParameters length:sizeof(theMazeMaskParameters) atIndex:BufferIndexCFMisc];

		if (itsNonuniformThreadGroupSizesAreAvailable)
		{
			//	Dispatch one thread per pixel, using the first strategy
			//	described in Apple's article
			//
			//		https://developer.apple.com/documentation/metal/calculating_threadgroup_and_grid_sizes
			//
			//	There's no reason theThreadgroupWidth must equal the threadExecutionWidth,
			//	but it's a convenient choice.
			//
			theThreadgroupWidth  = [itsMakeMazeMaskPipelineState threadExecutionWidth];			//	hardware-dependent constant (typically 32)
			theThreadgroupHeight = [itsMakeMazeMaskPipelineState maxTotalThreadsPerThreadgroup]	//	varies according to program resource needs
								 / theThreadgroupWidth;
			
			[theComputeEncoder dispatchThreads:	MTLSizeMake(thePixelsPerRow, thePixelsPerRow, 1)
						 threadsPerThreadgroup:	MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
		}
		else
		{
			//	Legacy method:
			//
			//	Use the second strategy described in Apple's article cited above.
			//	We must ensure that theThreadgroupWidth and theThreadgroupHeight
			//	exactly divide thePixelsPerRow.  The GPU's threadExecutionWidth
			//	is almost surely a power of two (and in fact on iOS might be 32
			//	in all cases that use this legacy method).
			//
			theThreadExecutionWidth = [itsMakeMazeMaskPipelineState threadExecutionWidth];
			if (IsPowerOfTwo((unsigned int)theThreadExecutionWidth))
			{
				theThreadgroupWidth  = theThreadExecutionWidth;
				theThreadgroupHeight = 1;
				
				//	Most likely those values will work fine.
				//	But just to be safe, let's make sure
				//	the theThreadgroupWidth doesn't exceed thePixelsPerRow.
				while (theThreadgroupWidth > thePixelsPerRow)
				{
					theThreadgroupWidth  /= 2;
					theThreadgroupHeight *= 2;
				}
				
				//	Just to be 100% safe, let's make sure we haven't
				//	pushed theThreadgroupHeight past thePixelsPerRow.
				if (theThreadgroupHeight > thePixelsPerRow)
					theThreadgroupHeight = thePixelsPerRow;
			}
			else
			{
				//	In the unlikely case that threadExecutionWidth isn't a power of two,
				//	fall back to the inefficient but safe strategy of processing the pixels
				//	one at a time.
				theThreadgroupWidth  = 1;
				theThreadgroupHeight = 1;
			}

			[theComputeEncoder dispatchThreadgroups: MTLSizeMake(
														thePixelsPerRow / theThreadgroupWidth,
														thePixelsPerRow / theThreadgroupHeight,
														1)
							  threadsPerThreadgroup: MTLSizeMake(theThreadgroupWidth, theThreadgroupHeight, 1)];
		}
		
		[theComputeEncoder endEncoding];

		[self generateMipmapsForTexture:theMaskTexture commandBuffer:theCommandBuffer];

		[theCommandBuffer commit];
	}
	else	//	! itsMazeIsValid
	{
		//	If the maze has not yet been initialized,
		//	make the whole alpha mask transparent.
		theMaskTexture = [self makeRGBATextureOfColor:(ColorP3Linear){0.0, 0.0, 0.0, 0.0} size:1];
	}
	
	return theMaskTexture;
}

- (id<MTLTexture>)makeCrosswordCharacterMask:(Char16)aCharacter
{
	Char16			theString[2];
	const Char16	*theFontName;
	unsigned int	theFontSize,
					theFontDescent;
	
	//	Even though the Torus Games' Crossword code
	//	works with lowercase characters internally,
	//	let's display them as uppercase (in languages
	//	which make a lowercase/uppercase distinction).
	aCharacter = ToUpperCase(aCharacter);

	theString[0] = aCharacter;
	theString[1] = 0;

	theFontName		= ChooseCrosswordCellFont();
	theFontSize		= (4 * CROSSWORD_CHARACTER_TEXTURE_SIZE) / 5;	//	integer division discards remainder
	theFontDescent	= CROSSWORD_CHARACTER_TEXTURE_SIZE - theFontSize;

	if (IsCurrentLanguage(u"vi"))
	{
		//	Vietnamese letters don't need a lot of space below the line,
		//	but some of them have two diacritics stacked on top,
		//	for example the uppercase ỗ = Ỗ, so we should allow space for them.
		//
		//		Note:  Xcode's code editor doesn't allow enough space
		//		to show both diacritics on the Ỗ, so you might need
		//		to copy-and-paste that character to a different
		//		text editor if you want to see them both.
		//
		theFontSize		= (3 * WORDSEARCH_CHARACTER_TEXTURE_SIZE) / 4;				//	integer division discards remainder
		theFontDescent	= 2*(WORDSEARCH_CHARACTER_TEXTURE_SIZE - theFontSize)/3;	//	integer division discards remainder
	}
	
	return [self createGreyscaleTextureWithString:	theString
											width:	CROSSWORD_CHARACTER_TEXTURE_SIZE
										   height:	CROSSWORD_CHARACTER_TEXTURE_SIZE
										 fontName:	theFontName
										 fontSize:	theFontSize
									  fontDescent:	theFontDescent
										 centered:	true
										   margin:	0];
}


- (id<MTLTexture>)makeWordSearchCharacterMask:(Char16)aCharacter
{
	Char16			theString[2];
	const Char16	*theFontName;
	unsigned int	theFontSize,
					theFontDescent;

	//	Even though the Torus Games' Word Search code
	//	works with lowercase characters internally,
	//	let's display them as uppercase (in languages
	//	which make a lowercase/uppercase distinction).
	aCharacter = ToUpperCase(aCharacter);
	
	theString[0] = aCharacter;
	theString[1] = 0;

	theFontName		= ChooseWordSearchCellFont();
	theFontSize		= (4 * WORDSEARCH_CHARACTER_TEXTURE_SIZE) / 5;	//	integer division discards remainder
	theFontDescent	= WORDSEARCH_CHARACTER_TEXTURE_SIZE - theFontSize;

	if (IsCurrentLanguage(u"vi"))
	{
		//	Vietnamese letters don't need a lot of space below the line,
		//	but some of them have two diacritics stacked on top,
		//	for example the uppercase ỗ = Ỗ, so we should allow space for them.
		//
		//		Note:  Xcode's code editor doesn't allow enough space
		//		to show both diacritics on the Ỗ, so you might need
		//		to copy-and-paste that character to a different
		//		text editor if you want to see them both.
		//
		theFontSize		= (3 * WORDSEARCH_CHARACTER_TEXTURE_SIZE) / 4;			//	integer division discards remainder
		theFontDescent	= (WORDSEARCH_CHARACTER_TEXTURE_SIZE - theFontSize)/2;	//	integer division discards remainder
	}

	return [self createGreyscaleTextureWithString:	theString
											width:	WORDSEARCH_CHARACTER_TEXTURE_SIZE
										   height:	WORDSEARCH_CHARACTER_TEXTURE_SIZE
										 fontName:	theFontName
										 fontSize:	theFontSize
									  fontDescent:	theFontDescent
										 centered:	true
										   margin:	0];
}

- (id<MTLTexture>)makeApplesNumeralMask:(Char16)aCharacter
{
	unsigned int	theCellSize,
					theFontSize,
					theFontDescent;
	const Char16	*theFontName;
	Char16			theString[2];

	theCellSize		= APPLES_CHARACTER_TEXTURE_SIZE;
	theFontSize		= (4 * theCellSize) / 5;	//	integer division discards remainder
	theFontDescent	= theCellSize - theFontSize;
	theFontName		= ChooseApplesNumeralFont();
	
	theString[0] = aCharacter;
	theString[1] = 0;
	
	return [self createGreyscaleTextureWithString:	theString
											width:	theCellSize
										   height:	theCellSize
										 fontName:	theFontName
										 fontSize:	theFontSize
									  fontDescent:	theFontDescent
										 centered:	true
										   margin:	0];
}


#pragma mark -
#pragma mark 2D game-independent rendering

- (void)encode2DCommandsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
				inflightDataBuffers:	(NSDictionary<NSString *, id> *)someInflightDataBuffers
						  modelData:	(ModelData *)md
{
	id<MTLBuffer>	the2DUniformBuffer,
					the2DCoveringTransformationBuffer,
					the2DSpritePlacementBuffer;
	NSUInteger		theNumCoveringTransformations;

	GEOMETRY_GAMES_ASSERT(
		! GameIs3D(md->itsGame),
		"Internal error:  got 3D game where 2D game was expected");

	the2DUniformBuffer					= [someInflightDataBuffers objectForKey:@"2D uniform buffer"				];
	the2DCoveringTransformationBuffer	= [someInflightDataBuffers objectForKey:@"2D covering transformation buffer"];
	the2DSpritePlacementBuffer			= [someInflightDataBuffers objectForKey:@"2D sprite placement buffer"		];
		//	Note:  the2DSpritePlacementBuffer will be nil if no sprites are present.
		//	There's no way to have a non-nil MTLBuffer of zero length.

	GEOMETRY_GAMES_ASSERT(
		[the2DCoveringTransformationBuffer length] % sizeof(TorusGames2DCoveringTransformation) == 0,
		"size of the2DCoveringTransformationBuffer is not an integer multiple of data size");

	theNumCoveringTransformations = [the2DCoveringTransformationBuffer length]
									/ sizeof(TorusGames2DCoveringTransformation);

	[aRenderEncoder setVertexBuffer:	the2DUniformBuffer
							 offset:	offsetof(TorusGames2DUniformData, itsWorldData)
							atIndex:	BufferIndexVFWorldData];

	[aRenderEncoder setVertexBuffer:	the2DCoveringTransformationBuffer
							 offset:	0
							atIndex:	BufferIndexVFCoveringTransformations];

	[aRenderEncoder setCullMode:MTLCullModeNone];

	//	Draw the background
	if (md->itsGame != GameNone)
	{
		[self encodeCommandsFor2DBackgroundWithEncoder:	aRenderEncoder
										 uniformBuffer:	the2DUniformBuffer
							numCoveringTransformations:	theNumCoveringTransformations];
	}

	//	Draw the game content
	switch (md->itsGame)
	{
		case Game2DIntro:
			[self	   encodeCommandsFor2DIntroWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DTicTacToe:
			[self  encodeCommandsFor2DTicTacToeWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DGomoku:
			[self	  encodeCommandsFor2DGomokuWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DMaze:
			[self		encodeCommandsFor2DMazeWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DCrossword:
			[self  encodeCommandsFor2DCrosswordWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DWordSearch:
			[self encodeCommandsFor2DWordSearchWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DJigsaw:
			[self	  encodeCommandsFor2DJigsawWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DChess:
			[self	   encodeCommandsFor2DChessWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DPool:
			[self		encodeCommandsFor2DPoolWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		case Game2DApples:
			[self	  encodeCommandsFor2DApplesWithEncoder:	aRenderEncoder
											 uniformBuffer:	the2DUniformBuffer
								numCoveringTransformations:	theNumCoveringTransformations
									 spritePlacementBuffer:	the2DSpritePlacementBuffer
												 modelData:	md];
			break;

		default:
			break;
	}
	
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	//	Draw the hand cursor
	[self encodeCommandsFor2DHandCursorWithEncoder:	aRenderEncoder
									 uniformBuffer:	the2DUniformBuffer
						numCoveringTransformations:	theNumCoveringTransformations
										 modelData:	md];
#endif
}

- (void)encodeCommandsFor2DBackgroundWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
{
	//	The caller should have already checked that
	//
	//		- the game is 2D, and
	//		- the game is not GameNone
	//
	//	but let's double-check that the background texture is non-nil,
	//	just to be safe.
	//
	if (itsTextures[Texture2DBackground] != nil)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DBackground];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setVertexBuffer:a2DUniformBuffer offset:offsetof(TorusGames2DUniformData, itsBackgroundTexReps) atIndex:BufferIndexVFMisc];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DBackground] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicRepeat atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DKleinAxesWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								  uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
{
	//	The caller should have already checked that
	//	- the game is 2D, and
	//	- the topology is Topology2DKlein

	//	Get set up
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DLineWithRGBATexture];
	[aRenderEncoder setVertexBuffer:a2DUniformBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a2DUniformBuffer offset:offsetof(TorusGames2DUniformData, itsKleinAxisTexReps) atIndex:BufferIndexVFMisc];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DKleinBottleAxis] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicRepeat atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:0
						atIndex:BufferIndexFFMisc];

	//	Draw one glide reflection axis
	[aRenderEncoder setVertexBufferOffset:offsetof(TorusGames2DUniformData, itsKleinAxisPlacementA) atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsKleinAxisColorA) atIndex:BufferIndexFFMisc];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];

	//	Draw the other glide reflection axis
	[aRenderEncoder setVertexBufferOffset:offsetof(TorusGames2DUniformData, itsKleinAxisPlacementB) atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsKleinAxisColorB) atIndex:BufferIndexFFMisc];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];
}

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

- (void)encodeCommandsFor2DHandCursorWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
									   modelData:	(ModelData *)md
{
	TextureIndex	theHandTextureIndex;

	//	The caller should have already checked that the game is 2D.

	if (md->its2DHandStatus != HandNone)
	{
		switch (md->its2DHandStatus)
		{
			case HandNone:	//	Suppress compiler warnings.  This case is already excluded by the enclosing "if" statement.
			case HandFree:
				theHandTextureIndex	= Texture2DHandFlat;
				break;

			case HandScroll:
			case HandDrag:
				theHandTextureIndex	= Texture2DHandGrab;
				break;
		}

		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsHandCursorPlacement)
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer
							offset:0
							atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[theHandTextureIndex] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

#endif	//	TORUS_GAMES_2D_MOUSE_INTERFACE


#pragma mark -
#pragma mark 2D game-specific rendering

- (void)encodeCommandsFor2DIntroWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							  uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					  spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								  modelData:	(ModelData *)md
{
	unsigned int	i;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DIntroSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the sprites in reverse order.
	//	The net effect is to draw the shells, then the crab, then finally the flounder.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (i = 4; i-- > 0; )
	{
		[aRenderEncoder setVertexBufferOffset:(i * sizeof(TorusGames2DSpritePlacementMatrix)) atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DIntroSpriteBaseIndex + i] atIndex:TextureIndexFF];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DTicTacToeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								  uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						  spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
									  modelData:	(ModelData *)md
{
	TicTacToePlayer	(*theBoard)[3];
	bool			theWinLineIsPresent;
	unsigned int	h,
					v;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DTicTacToeSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Get relevant information from the ModelData
	theBoard			= md->itsGameOf.TicTacToe2D.itsBoard;
	theWinLineIsPresent	= (md->itsGameIsOver && ! md->itsFlashFlag);

	//	Draw the cell boundaries
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DTicTacToeGrid] atIndex:TextureIndexFF];
	for (h = 0; h < 3; h++)
	{
		for (v = 0; v < 3; v++)
		{
			[aRenderEncoder	setVertexBufferOffset:	((3*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
										  atIndex:	BufferIndexVFPlacement];
			
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
	}

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the cell contents

	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	
	//		PlayerX
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DTicTacToeMarkerX] atIndex:TextureIndexFF];
	for (h = 0; h < 3; h++)
	{
		for (v = 0; v < 3; v++)
		{
			if (theBoard[h][v] == PlayerX)
			{
				[aRenderEncoder	setVertexBufferOffset:	((3*3 + 3*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//		PlayerO
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DTicTacToeMarkerO] atIndex:TextureIndexFF];
	for (h = 0; h < 3; h++)
	{
		for (v = 0; v < 3; v++)
		{
			if (theBoard[h][v] == PlayerO)
			{
				[aRenderEncoder	setVertexBufferOffset:	((3*3 + 3*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}
	
	//	Draw the win line, if present
#ifdef SHAPE_OF_SPACE_TICTACTOE
	if (false)	//	suppress the win line for Shape of Space Figure 2.2
#else
	if (theWinLineIsPresent)
#endif
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(3*3 + 3*3) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DTicTacToeWinLine] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicRepeat atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DGomokuWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					   spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								   modelData:	(ModelData *)md
{
	GomokuIntersection	(*theIntersections)[GOMOKU_SIZE];
	bool				theWinLineIsPresent;
	unsigned int		h,
						v;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DGomokuSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Get relevant information from the ModelData.
	theIntersections	= md->itsGameOf.Gomoku2D.itsBoard.itsIntersections;

	//	If a player (either human or computer) has won the game,
	//	draw the winning five-in-a-row.
	//
	//	If sound is disabled, flash the winning five-in-a-row 
	//		to make the win obvious.
	//	If sound is enabled, it alone will make the win obvious,
	//		so suppress the flashing to keep the game calmer.
	//
	theWinLineIsPresent	= ( md->itsGameOf.Gomoku2D.itsGameStatus == GomokuGameWin
						 && (gPlaySounds || ! md->itsFlashFlag) );

	//	Draw the win line, if present
	if (theWinLineIsPresent)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DLineWithEndcapsAndMask];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(GOMOKU_SIZE*GOMOKU_SIZE) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DLineWithEndcapsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DGomokuWinLine] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsGomokuWinLineColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_LINE_WITH_ENDCAPS_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}

	//	Draw the empty cells before drawing the possible glide reflection axes
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	index to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DGomokuGrid] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			if (theIntersections[h][v].itsStone == GomokuPlayerNone)
			{
				[aRenderEncoder setVertexBufferOffset:	(GOMOKU_SIZE*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the non-empty cells after drawing the possible glide reflection axes

	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];

	//		black stones
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DGomokuStoneBlack] atIndex:TextureIndexFF];
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			if (theIntersections[h][v].itsStone == GomokuPlayerBlack)
			{
				[aRenderEncoder setVertexBufferOffset:	(GOMOKU_SIZE*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//		white stones
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DGomokuStoneWhite] atIndex:TextureIndexFF];
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			if (theIntersections[h][v].itsStone == GomokuPlayerWhite)
			{
				[aRenderEncoder setVertexBufferOffset:	(GOMOKU_SIZE*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}
}

- (void)encodeCommandsFor2DMazeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							 uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					 spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								 modelData:	(ModelData *)md
{
	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DMazeSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}
	
	//	Draw the maze walls
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:(0 * sizeof(TorusGames2DSpritePlacementMatrix))
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DMazeMaze] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsMazeWallColor)
						atIndex:BufferIndexFFMisc];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];

	//	Draw the mouse and the cheese
	//
	//		Draw the cheese before the mouse, so the mouse
	//		stacks on top of it at the end of the game.
	
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];

	//		cheese
	if ( ! md->itsGameIsOver
	 || md->itsFlashFlag)
	{
		[aRenderEncoder setVertexBufferOffset:(2 * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DMazeCheese] atIndex:TextureIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}

	//		mouse
	[aRenderEncoder setVertexBufferOffset:(1 * sizeof(TorusGames2DSpritePlacementMatrix))
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DMazeMouse] atIndex:TextureIndexFF];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];
}

- (void)encodeCommandsFor2DCrosswordWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								  uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						  spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
									  modelData:	(ModelData *)md
{
	unsigned int	thePuzzleSize,
					theHotCellH,
					theHotCellV;
	Char16			(*theBoard)[MAX_CROSSWORD_SIZE],
					(*theSolution)[MAX_CROSSWORD_SIZE];
	bool			(*theHotWord)[MAX_CROSSWORD_SIZE],
					theGameIsOver;
	unsigned int	h,
					v,
					theTextureIndex;

	//	Get relevant information from the ModelData
	thePuzzleSize	= md->itsGameOf.Crossword2D.itsPuzzleSize;
	theHotCellH		= md->itsGameOf.Crossword2D.itsHotCellH;
	theHotCellV		= md->itsGameOf.Crossword2D.itsHotCellV;
	theBoard		= md->itsGameOf.Crossword2D.itsBoard;
	theSolution		= md->itsGameOf.Crossword2D.itsSolution;
	theHotWord		= md->itsGameOf.Crossword2D.itsHotWord;
	theGameIsOver	= md->itsGameIsOver;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DCrosswordSprites(thePuzzleSize) * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");
	
	//	For a few seconds at the end of each puzzle,
	//	while the victory sound is playing, overdraw
	//	the Crossword's usual white background with pure green.
	//
	//		Technical note:  This approach works better than requesting
	//		a white or green background color in clearColorWithModelData,
	//		because
	//
	//			- all 2D games treat the usual background the same way
	//				(the Crossword uses a pure white texture)
	//
	//			- all 2D games treat the clear color the same way
	//				(always pure transparent)
	//
	//			- when zooming between the Basic and Repeating views,
	//				the unused margin region is transparent (in effect black)
	//				not white, so the user may easily see that
	//				the margin isn't part of the torus game board.
	//
	
	if (md->itsFlashFlag)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColor];
		[aRenderEncoder	setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(thePuzzleSize*thePuzzleSize + 1) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:nil atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:nil atIndex:SamplerIndexFF];	//	On iOS 10.0, crashes with EXC_BAD_ACCESS;  OK on iOS 12.4.8
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsCrosswordFlashColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
	
	//	Draw black cells.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColor];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:nil atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:nil atIndex:SamplerIndexFF];	//	On iOS 10.0, crashes with EXC_BAD_ACCESS;  OK on iOS 12.4.8
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsCrosswordGridColor)
						atIndex:BufferIndexFFMisc];
	for (h = 0; h < thePuzzleSize; h++)
	{
		for (v = 0; v < thePuzzleSize; v++)
		{
			if (theSolution[h][v] == L'*')	//	black cell?
			{
				[aRenderEncoder	setVertexBufferOffset:	(thePuzzleSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

#ifdef MAKE_GAME_CHOICE_ICONS
	//	Suppress hot word coloring for the game choice icon.
#else
	//	Draw backgrounds for hot letter and hot word,
	//	if the game isn't yet over.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColor];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:nil atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:nil atIndex:SamplerIndexFF];	//	On iOS 10.0, crashes with EXC_BAD_ACCESS;  OK on iOS 12.4.8
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexFFMisc];
	if ( ! theGameIsOver )
	{
		for (h = 0; h < thePuzzleSize; h++)
		{
			for (v = 0; v < thePuzzleSize; v++)
			{
				if (theHotWord[h][v])	//	is this cell part of the hot word?
				{
					[aRenderEncoder	  setVertexBufferOffset:	(thePuzzleSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
													atIndex:	BufferIndexVFPlacement];

					[aRenderEncoder setFragmentBufferOffset:	((h == theHotCellH && v == theHotCellV) ?
																	offsetof(TorusGames2DUniformData, itsCrosswordHotCellColor) :
																	offsetof(TorusGames2DUniformData, itsCrosswordHotWordColor)
																)
													atIndex:	BufferIndexFFMisc];
					
					[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
									   vertexStart:	0
									   vertexCount:	NUM_SQUARE_VERTICES
									 instanceCount:	aNumCoveringTransformations];
				}
			}
		}
	}
#endif
	
	//	Draw the grid.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DGrid];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a2DUniformBuffer offset:offsetof(TorusGames2DUniformData, itsCrosswordGridTexReps) atIndex:BufferIndexVFMisc];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsCrosswordGridLineTransitionZoneWidthInverse)
						atIndex:BufferIndexFFMisc];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];

	//	Draw glide reflection axes, if present.
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the characters.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsCrosswordCharacterColor)
						atIndex:BufferIndexFFMisc];
	for (h = 0; h < thePuzzleSize; h++)
	{
		for (v = 0; v < thePuzzleSize; v++)
		{
			if (theSolution[h][v] != L'*'	//	not a black cell
			 &&    theBoard[h][v] != L' ')	//	not an empty cell
			{
				[aRenderEncoder	setVertexBufferOffset:	(thePuzzleSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];

				theTextureIndex = Texture2DCrosswordFirstCell + h*thePuzzleSize + v;
				GEOMETRY_GAMES_ASSERT(itsTextures[theTextureIndex] != nil, "missing Crossword character texture");
				[aRenderEncoder setFragmentTexture:	itsTextures[theTextureIndex]
										   atIndex:	TextureIndexFF];
				
				[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
	//	Don't draw the word direction arrow for a screenshot.
	//	For the screenshot we want an already filled-in word to be highlighted,
	//	without the distraction of the direction arrow.
	//	For the game choice icon, we want an empty board.
#else
	//	Draw the word-direction arrow,
	//	if the game isn't yet over.
	if ( ! theGameIsOver )
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
		[aRenderEncoder	setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(thePuzzleSize*thePuzzleSize) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DCrosswordArrow] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsCrosswordArrowColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
#endif
	
	//	Draw the pending character, if any.
	if (itsTextures[Texture2DCrosswordPendingCharacter] != nil		//	is a pending character present?
	 && theHotCellH < thePuzzleSize && theHotCellV < thePuzzleSize)	//	should never fail
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
		//	Technical note:  itsHotCellFlip should agree with itsBoardFlips[theHotCellH][theHotCellV].
		//	In any case, draw the pending character using the latter, which is already available
		//	in aSpritePlacementBuffer.
		[aRenderEncoder	setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(thePuzzleSize*theHotCellH + theHotCellV) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DCrosswordPendingCharacter] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsCrosswordPendingCharacterColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DWordSearchWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
					  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
						   spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
									   modelData:	(ModelData *)md
{
	unsigned int	thePuzzleSize,
					theNumLinesFound;
	bool			theWordSelectionIsPending,
					theGameIsOver,
					theFlashFlag;
	unsigned int	h,
					v,
					i;
	unsigned int	theTextureIndex;

	//	Get relevant information from the ModelData
	thePuzzleSize				= md->itsGameOf.WordSearch2D.itsPuzzleSize;
	theNumLinesFound			= md->itsGameOf.WordSearch2D.itsNumLinesFound;
	theWordSelectionIsPending	= md->itsGameOf.WordSearch2D.itsWordSelectionIsPending;
	theGameIsOver				= md->itsGameIsOver;
	theFlashFlag				= md->itsFlashFlag;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DWordSearchSprites(thePuzzleSize, theNumLinesFound) * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present.
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}
	
	//	Mark all previously found words, and also the pending word if present.
	//	Exception:  Don't mark words while flashing at the end of a game.
	if ( ! (theGameIsOver && theFlashFlag) )
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DLineWithEndcapsRGBATexture];
		[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DLineWithEndcapsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DWordSearchMark] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexFFMisc];
		for (i = 0; i < theNumLinesFound; i++)
		{
			[aRenderEncoder   setVertexBufferOffset:	(thePuzzleSize*thePuzzleSize + i) * sizeof(TorusGames2DSpritePlacementMatrix)
											atIndex:	BufferIndexVFPlacement];

			[aRenderEncoder setFragmentBufferOffset:	offsetof(TorusGames2DUniformData, itsWordSearchMarkedWordColors[i])
											atIndex:	BufferIndexFFMisc];

			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_LINE_WITH_ENDCAPS_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
		if (theWordSelectionIsPending)
		{
			[aRenderEncoder   setVertexBufferOffset:	(thePuzzleSize*thePuzzleSize + theNumLinesFound) * sizeof(TorusGames2DSpritePlacementMatrix)
											atIndex:	BufferIndexVFPlacement];

			[aRenderEncoder setFragmentBufferOffset:	offsetof(TorusGames2DUniformData, itsWordSearchPendingWordColor)
											atIndex:	BufferIndexFFMisc];

			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_LINE_WITH_ENDCAPS_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
	}

	//	Draw the characters.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsWordSearchCharacterColor)
						atIndex:BufferIndexFFMisc];
	for (h = 0; h < thePuzzleSize; h++)
	{
		for (v = 0; v < thePuzzleSize; v++)
		{
			[aRenderEncoder setVertexBufferOffset:	(thePuzzleSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
										  atIndex:	BufferIndexVFPlacement];

			theTextureIndex = Texture2DWordSearchFirstCell + h*thePuzzleSize + v;
			GEOMETRY_GAMES_ASSERT(itsTextures[theTextureIndex] != nil, "missing Word Search character texture");
			[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
			
			[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
	}

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
	//	Draw the selection's hot spot, if relevant.
	if (theWordSelectionIsPending)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(thePuzzleSize*thePuzzleSize + theNumLinesFound + 1) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DWordSearchHotSpot] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsWordSearchHotPointColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
#endif
}

- (void)encodeCommandsFor2DJigsawWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					   spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								   modelData:	(ModelData *)md
{
	unsigned int	thePuzzleSize,
					theNumPieces,
					i;

	//	Get relevant information from the ModelData
	thePuzzleSize	= md->itsGameOf.Jigsaw2D.itsSize;
	theNumPieces	= thePuzzleSize * thePuzzleSize;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DJigsawSprites(thePuzzleSize) * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}
	
	//	Draw the puzzle pieces
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTextureSubset];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a2DUniformBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFMisc];
	[aRenderEncoder setFragmentTexture:itsTextures[md->itsGameIsOver ?
													Texture2DJigsawCollageWithoutBorders :
													Texture2DJigsawCollageWithBorders]
												atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (i = 0; i < theNumPieces; i++)
	{
		[aRenderEncoder setVertexBufferOffset:	i * sizeof(TorusGames2DSpritePlacementMatrix)
									  atIndex:	BufferIndexVFPlacement];

		[aRenderEncoder setVertexBufferOffset:	offsetof(TorusGames2DUniformData, itsJigsawTexturePlacement[i])
									  atIndex:	BufferIndexVFMisc];

		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DChessWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							  uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					  spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								  modelData:	(ModelData *)md
{
	unsigned int	h,
					v,
					theSquare;
	bool			thePieceIsInMotion;
	TextureIndex	theTextureIndexBase;
	unsigned int	theTextureIndexOffset;
	TextureIndex	theTextureIndex;
	unsigned int	theDragPiece;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DChessSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	If the user has attempted to drag a piece of the wrong color,
	//	draw halos under the pieces of the right color.
	if (md->its2DHandStatus == HandDrag && ! md->itsGameOf.Chess2D.itsUserMoveIsPending)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
		[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessHalo] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsChessWhoseTurnHaloColor)
							atIndex:BufferIndexFFMisc];
		for (h = 0; h < 8; h++)
		{
			for (v = 0; v < 8; v++)
			{
				theSquare = md->itsGameOf.Chess2D.itsSquares[h][v];

				if (theSquare != CHESS_EMPTY_SQUARE
				 && (theSquare & CHESS_COLOR_MASK) == md->itsGameOf.Chess2D.itsWhoseTurn)
				{
					[aRenderEncoder setVertexBufferOffset:	(8*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
												  atIndex:	BufferIndexVFPlacement];

					[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
									   vertexStart:	0
									   vertexCount:	NUM_SQUARE_VERTICES
									 instanceCount:	aNumCoveringTransformations];
				}
			}
		}
	}

	//	Draw the opaque pieces
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (h = 0; h < 8; h++)
	{
		for (v = 0; v < 8; v++)
		{
			theSquare			= md->itsGameOf.Chess2D.itsSquares[h][v];
			thePieceIsInMotion	=
				(	(	md->itsGameOf.Chess2D.itsUserMoveIsPending
					 || md->itsGameOf.Chess2D.itsComputerMoveIsPending )
				 && md->itsGameOf.Chess2D.itsMoveStartH == h
				 && md->itsGameOf.Chess2D.itsMoveStartV == v );

			if ( theSquare != CHESS_EMPTY_SQUARE
			 && ! thePieceIsInMotion )
			{
				//	aSpritePlacementBuffer already contains the correct cell placements
				//	including the possible flips in the Klein bottle, but...
				[aRenderEncoder setVertexBufferOffset:	(8*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
											  atIndex:	BufferIndexVFPlacement];
				
				//	...we must still select the appropriate texture.

				theTextureIndexBase		= ((theSquare & CHESS_COLOR_MASK) == CHESS_WHITE ?
											Texture2DChessWhitePieceBaseIndex :
											Texture2DChessBlackPieceBaseIndex );
				theTextureIndexOffset	= (theSquare & CHESS_PIECE_MASK) - 1;
				theTextureIndex			= theTextureIndexBase + theTextureIndexOffset;

				[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];

				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//	Draw the check, if any.
	//	Note that we draw it after drawing the pieces,
	//	so that it will sit on top of (not underneath) neighboring pieces.
	if (md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaH != 0
	 || md->itsGameOf.Chess2D.itsCheckThreat.itsDeltaV != 0)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];

		//	If the king has been checkmated or stalemated,
		//	don't draw the "target" graphic underneath,
		//	but instead just wait for the checkmate or stalemate
		//	graphic to go on top.
		if ( ! md->itsGameIsOver )
		{
			//	Draw the "target" graphic.
			[aRenderEncoder setVertexBufferOffset:((8*8 + 2) * sizeof(TorusGames2DSpritePlacementMatrix))
								atIndex:BufferIndexVFPlacement];
			[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessTarget] atIndex:TextureIndexFF];
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];

			//	If the threatened square is non-empty,
			//	redraw its contents on top of the "target" sprite.
			//
			//		Note:  When the user attempts to move the king
			//		to a threatened square, we leave the king in its previous square
			//		and show the threat using whatever other piece (if any)
			//		may be present there.
			//
			h			= md->itsGameOf.Chess2D.itsCheckThreat.itsFinalH;
			v			= md->itsGameOf.Chess2D.itsCheckThreat.itsFinalV;
			theSquare	= md->itsGameOf.Chess2D.itsSquares[h][v];
			if (theSquare != CHESS_EMPTY_SQUARE)
			{
				theTextureIndexBase		= ((theSquare & CHESS_COLOR_MASK) == CHESS_WHITE ?
											Texture2DChessWhitePieceBaseIndex :
											Texture2DChessBlackPieceBaseIndex );
				theTextureIndexOffset	= (theSquare & CHESS_PIECE_MASK) - 1;
				theTextureIndex			= theTextureIndexBase + theTextureIndexOffset;

				[aRenderEncoder setVertexBufferOffset:((8*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
									atIndex:BufferIndexVFPlacement];
				[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
				[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}

		//	Draw the attacking piece's arrow no matter what.
		[aRenderEncoder setVertexBufferOffset:((8*8 + 1) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessArrow] atIndex:TextureIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

		//	Redraw the attacking piece on top of the arrow.

		h			= md->itsGameOf.Chess2D.itsCheckThreat.itsStartH;
		v			= md->itsGameOf.Chess2D.itsCheckThreat.itsStartV;
		theSquare	= md->itsGameOf.Chess2D.itsSquares[h][v];

		theTextureIndexBase		= ((theSquare & CHESS_COLOR_MASK) == CHESS_WHITE ?
									Texture2DChessWhitePieceBaseIndex :
									Texture2DChessBlackPieceBaseIndex );
		theTextureIndexOffset	= (theSquare & CHESS_PIECE_MASK) - 1;
		theTextureIndex			= theTextureIndexBase + theTextureIndexOffset;

		[aRenderEncoder setVertexBufferOffset:((8*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}

	//	Draw the checkmate or stalemate graphic if the game is over.
	if (md->itsGameIsOver)
	{
		if (md->itsGameOf.Chess2D.itsCheckmateFlag)
			theTextureIndex = Texture2DChessCheckmate;
		else
			theTextureIndex = Texture2DChessStalemate;
		
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(8*8 + 3) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}

	//	Draw the piece-in-motion, if any.
	if (md->itsGameOf.Chess2D.itsUserMoveIsPending
	 || md->itsGameOf.Chess2D.itsComputerMoveIsPending)
	{
		h				= md->itsGameOf.Chess2D.itsMoveStartH;
		v				= md->itsGameOf.Chess2D.itsMoveStartV;
		theDragPiece	= md->itsGameOf.Chess2D.itsSquares[h][v];

		theTextureIndexBase		= ((theDragPiece & CHESS_COLOR_MASK) == CHESS_WHITE ?
									Texture2DChessWhitePieceBaseIndex :
									Texture2DChessBlackPieceBaseIndex );
		theTextureIndexOffset	= (theDragPiece & CHESS_PIECE_MASK) - 1;
		theTextureIndex			= theTextureIndexBase + theTextureIndexOffset;
		
		//	Draw a "ghost image" in the piece-in-motion's starting square.
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndRGBATexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(8*h + v) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:offsetof(TorusGames2DUniformData, itsChessGhostColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
		//	Draw a halo under the piece-in-motion.
		if ( ! md->itsHumanVsComputer || md->itsGameOf.Chess2D.itsWhoseTurn == CHESS_WHITE)
		{
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
			[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
									 offset:	(8*8 + 0) * sizeof(TorusGames2DSpritePlacementMatrix)
									atIndex:	BufferIndexVFPlacement];
			[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
			[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessHalo] atIndex:TextureIndexFF];
			[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
			[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
								offset:offsetof(TorusGames2DUniformData, itsChessDragPieceHaloColor)
								atIndex:BufferIndexFFMisc];
			[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
#endif

		//	Draw the piece-in-motion with full opacity.
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(8*8 + 0) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[theTextureIndex] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}

	//	If the computer is thinking about its move,
	//	draw the progress indicator dial.
	if (md->itsSimulationStatus == Simulation2DChessChooseComputerMove
	 && md->itsSimulationElapsedTime > 1.0	//	suppress progress meter for short waits
	 && md->itsProgressMeterIsActive)
	{
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
		[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
							offset:0	//	offset to be set below
							atIndex:BufferIndexFFMisc];

		//	base
		[aRenderEncoder setVertexBufferOffset:((8*8 + 4 + 0) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessDialBase] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialBaseColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

		//	sweep A
		[aRenderEncoder setVertexBufferOffset:((8*8 + 4 + 1) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessDialSweep] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialSweepColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

		//	sweep B
		[aRenderEncoder setVertexBufferOffset:((8*8 + 4 + 2) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessDialSweep] atIndex:TextureIndexFF];
		if (md->itsProgressMeter <= 0.5)
			[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialBaseColor )
								atIndex:BufferIndexFFMisc];
		else
			[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialSweepColor)
								atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
		
		//	rim
		[aRenderEncoder setVertexBufferOffset:((8*8 + 4 + 3) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessDialRim] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialRimColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
		
		//	arrow
		[aRenderEncoder setVertexBufferOffset:((8*8 + 4 + 4) * sizeof(TorusGames2DSpritePlacementMatrix))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DChessDialArrow] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentBufferOffset:offsetof(TorusGames2DUniformData, itsChessDialArrowColor)
							atIndex:BufferIndexFFMisc];
		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];
	}
}

- (void)encodeCommandsFor2DPoolWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							 uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					 spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								 modelData:	(ModelData *)md
{
	unsigned int	i;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DPoolSprites() * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the pocket
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
							 offset:	(NUM_POOL_BALLS + 0) * sizeof(TorusGames2DSpritePlacementMatrix)
							atIndex:	BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DPoolPocket] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
					   vertexStart:	0
					   vertexCount:	NUM_SQUARE_VERTICES
					 instanceCount:	aNumCoveringTransformations];

	//	Draw the balls
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			[aRenderEncoder setVertexBufferOffset:(i * sizeof(TorusGames2DSpritePlacementMatrix))
								atIndex:BufferIndexVFPlacement];

			[aRenderEncoder setFragmentTexture:itsTextures[Texture2DPoolBallBaseIndex + i] atIndex:TextureIndexFF];

			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
	}

	//	Draw the cue stick if necessary.
	if ( ! md->itsGameIsOver
	 && md->itsGameOf.Pool2D.itsCueStatus != PoolCueNone)
	{
		//	stick
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(NUM_POOL_BALLS + 1) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DPoolStick] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

		//	line of sight
		[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
		[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
								 offset:	(NUM_POOL_BALLS + 2) * sizeof(TorusGames2DSpritePlacementMatrix)
								atIndex:	BufferIndexVFPlacement];
		[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DPoolLineOfSight] atIndex:TextureIndexFF];
		[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
		[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	NUM_SQUARE_VERTICES
						 instanceCount:	aNumCoveringTransformations];

#if defined(TORUS_GAMES_2D_TOUCH_INTERFACE) && ! defined(GAME_CONTENT_FOR_SCREENSHOT)
		//	hot spot
		if (md->itsGameOf.Pool2D.itsCueStatus == PoolCueActive)
		{
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
			[aRenderEncoder setVertexBuffer:	aSpritePlacementBuffer
									 offset:	(NUM_POOL_BALLS + 3) * sizeof(TorusGames2DSpritePlacementMatrix)
									atIndex:	BufferIndexVFPlacement];
			[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
			[aRenderEncoder setFragmentTexture:itsTextures[Texture2DPoolHotSpot] atIndex:TextureIndexFF];
			[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
			[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
								offset:offsetof(TorusGames2DUniformData, itsPoolHotSpotColor)
								atIndex:BufferIndexFFMisc];
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							   vertexStart:	0
							   vertexCount:	NUM_SQUARE_VERTICES
							 instanceCount:	aNumCoveringTransformations];
		}
#endif
	}
}

- (void)encodeCommandsFor2DApplesWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							   uniformBuffer:	(id<MTLBuffer>)a2DUniformBuffer
				  numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
					   spritePlacementBuffer:	(id<MTLBuffer>)aSpritePlacementBuffer
								   modelData:	(ModelData *)md
{
	unsigned int	theBoardSize;
	AppleStatus		(*theBoard)[MAX_APPLE_BOARD_SIZE];
	unsigned int	h,
					v,
					theNumNbrWorms;	//	number of neighboring cells that contain worms

	//	Get relevant information from the ModelData
	theBoardSize	= md->itsGameOf.Apples2D.itsBoardSize;
	theBoard		= md->itsGameOf.Apples2D.itsBoard;

	GEOMETRY_GAMES_ASSERT(
		[aSpritePlacementBuffer length] == GetNum2DApplesSprites(theBoardSize) * sizeof(TorusGames2DSpritePlacementMatrix),
		"Sprite placement buffer has wrong size");

	//	Draw glide reflection axes, if present
	if (md->itsTopology == Topology2DKlein)
	{
		[self encodeCommandsFor2DKleinAxesWithEncoder:	aRenderEncoder
										uniformBuffer:	a2DUniformBuffer
						   numCoveringTransformations:	aNumCoveringTransformations];

	}

	//	Draw the apples.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DApplesApple] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (h = 0; h < theBoardSize; h++)
	{
		for (v = 0; v < theBoardSize; v++)
		{
			if ( ! (theBoard[h][v] & APPLE_EXPOSED_MASK) )
			{
				[aRenderEncoder setVertexBufferOffset:((theBoardSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
									atIndex:BufferIndexVFPlacement];

				[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//	Draw the happy/yucky faces.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[md->itsGameIsOver ? Texture2DApplesFaceHappy : Texture2DApplesFaceYucky] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (h = 0; h < theBoardSize; h++)
	{
		for (v = 0; v < theBoardSize; v++)
		{
			if ( ! (theBoard[h][v] & APPLE_EXPOSED_MASK)
			 &&    (theBoard[h][v] & APPLE_MARKED_MASK ) )
			{
				[aRenderEncoder setVertexBufferOffset:	( (theBoardSize*theBoardSize + theBoardSize*h + v)
															* sizeof(TorusGames2DSpritePlacementMatrix) )
											  atIndex:	BufferIndexVFPlacement];

				[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//	Draw the worms.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithTexture];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentTexture:itsTextures[Texture2DApplesWorm] atIndex:TextureIndexFF];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	for (h = 0; h < theBoardSize; h++)
	{
		for (v = 0; v < theBoardSize; v++)
		{
			if ((theBoard[h][v] & APPLE_EXPOSED_MASK )
			 && (theBoard[h][v] & APPLE_HAS_WORM_MASK))
			{
				[aRenderEncoder setVertexBufferOffset:	((theBoardSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
											  atIndex:	BufferIndexVFPlacement];
				
				[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
								   vertexStart:	0
								   vertexCount:	NUM_SQUARE_VERTICES
								 instanceCount:	aNumCoveringTransformations];
			}
		}
	}

	//	Draw the numerals.
	//	To minimize texture changes, first draw all the 1's, then all the 2's, etc.
	[aRenderEncoder setRenderPipelineState:itsRenderPipelineState2DSpriteWithColorAndMask];
	[aRenderEncoder setVertexBuffer:aSpritePlacementBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:its2DSquareVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setFragmentSamplerState:itsTextureSamplerIsotropicClampToEdge atIndex:SamplerIndexFF];
	[aRenderEncoder setFragmentBuffer:a2DUniformBuffer
						offset:offsetof(TorusGames2DUniformData, itsApplesNumeralColor)
						atIndex:BufferIndexFFMisc];
	for (theNumNbrWorms = 1; theNumNbrWorms <= 8; theNumNbrWorms++)	//	skip 0
	{
		[aRenderEncoder setFragmentTexture:itsTextures[Texture2DApplesNumeralBaseIndex + theNumNbrWorms] atIndex:TextureIndexFF];

		for (h = 0; h < theBoardSize; h++)
		{
			for (v = 0; v < theBoardSize; v++)
			{
				if (   (theBoard[h][v] & APPLE_EXPOSED_MASK )
				 &&  ! (theBoard[h][v] & APPLE_HAS_WORM_MASK)
				 &&    (theBoard[h][v] & APPLE_NBR_COUNT_MASK) == theNumNbrWorms)
				{
					[aRenderEncoder setVertexBufferOffset:	((theBoardSize*h + v) * sizeof(TorusGames2DSpritePlacementMatrix))
												  atIndex:	BufferIndexVFPlacement];
					
					[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
									   vertexStart:	0
									   vertexCount:	NUM_SQUARE_VERTICES
									 instanceCount:	aNumCoveringTransformations];
				}
			}
		}
	}
}


#pragma mark -
#pragma mark 3D game-independent rendering

- (void)encode3DCommandsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
				inflightDataBuffers:	(NSDictionary<NSString *, id> *)someInflightDataBuffers
						  modelData:	(ModelData *)md
{
	id<MTLBuffer>							the3DUniformBuffer,
											the3DPlainCoveringTransformationBuffer,
											the3DReflectingCoveringTransformationBuffer,
											the3DPolyhedronPlacementBuffer;
	NSUInteger								theMaxNumPlainCoveringTransformations,
											theMaxNumReflectingCoveringTransformations,
											theNumPlainCoveringTransformations,
											theNumReflectingCoveringTransformations;
	TorusGames3DCoveringTransformation		*thePlainCoveringTransformations,
											*theReflectingCoveringTransformations;
	unsigned int							i;
	double									theTilingIntoFrameCellDeterminant;
	MTLWinding								theWindingForPlainCoveringTransformations,
											theWindingForReflectingCoveringTransformations;
	unsigned int							theSolidBufferLength,					//	in placements, not bytes
											theRectangularSliceBufferLength,		//	in placements, not bytes
											theCircularSliceBufferLength,			//	in placements, not bytes
											theClippedEllipticalSliceBufferLength,	//	in placements, not bytes
											theSolidBufferOffset,					//	in placements, not bytes
											theRectangularSliceBufferOffset,		//	in placements, not bytes
											theCircularSliceBufferOffset,			//	in placements, not bytes
											theClippedEllipticalSliceBufferOffset;	//	in placements, not bytes
	TorusGames3DPolyhedronPlacementAsSIMD	*thePlacementBufferContents;
	unsigned int							theNumCircularSliceVertices;
	id<MTLBuffer>							theCircularSliceVertexBuffer;

	GEOMETRY_GAMES_ASSERT(
		GameIs3D(md->itsGame),
		"Internal error:  got 2D game where 3D game was expected");

	the3DUniformBuffer							= [someInflightDataBuffers objectForKey:@"3D uniform buffer"							];
	the3DPlainCoveringTransformationBuffer		= [someInflightDataBuffers objectForKey:@"3D plain covering transformation buffer"		];
	the3DReflectingCoveringTransformationBuffer	= [someInflightDataBuffers objectForKey:@"3D reflecting covering transformation buffer"	];
	the3DPolyhedronPlacementBuffer				= [someInflightDataBuffers objectForKey:@"3D polyhedron placement buffer"				];
		//	Note:  the3DPolyhedronPlacementBuffer will be nil if no polyhedra are present.
		//	There's no way to have a non-nil MTLBuffer of zero length.

	GEOMETRY_GAMES_ASSERT(
		[the3DPlainCoveringTransformationBuffer      length] % sizeof(TorusGames3DCoveringTransformation) == 0,
		"size of   plain    covering transformation buffer is not an integer multiple of data size");
	GEOMETRY_GAMES_ASSERT(
		[the3DReflectingCoveringTransformationBuffer length] % sizeof(TorusGames3DCoveringTransformation) == 0,
		"size of reflecting covering transformation buffer is not an integer multiple of data size");

	theMaxNumPlainCoveringTransformations		= [the3DPlainCoveringTransformationBuffer      length] / sizeof(TorusGames3DCoveringTransformation);
	theMaxNumReflectingCoveringTransformations	= [the3DReflectingCoveringTransformationBuffer length] / sizeof(TorusGames3DCoveringTransformation);
	
	//	The covering transformation buffers get reallocated only rarely,
	//	for example when changing between ViewBasicLarge and ViewRepeating.
	//	To accommodate small but frequent changes in the number of covering transformations
	//	(for example when scrolling), the method
	//
	//		-write3DCoveringTransformationsIntoBuffersPlain:reflecting:modelData:
	//
	//	uses whatever space it needs and fills any unused buffer space with zero matrices,
	//	which may be reliably tested for by looking at the [3][3] entry.
	//	The [3][3] entry will always be 1.0 for a valid covering transformation,
	//	and will always be 0.0 for a filler matrix.
	//
	//	Count how many covering transformations are in use in the plain and reflecting transformation buffers.
	//	The in-use covering transformations always appear contiguously at the beginning of each buffer.

	thePlainCoveringTransformations			= [the3DPlainCoveringTransformationBuffer contents];
	theNumPlainCoveringTransformations		= 0;
	for (i = 0; i < theMaxNumPlainCoveringTransformations; i++)
		if (thePlainCoveringTransformations[i].columns[3][3] == 1.0)
			theNumPlainCoveringTransformations++;

	theReflectingCoveringTransformations	= [the3DReflectingCoveringTransformationBuffer contents];
	theNumReflectingCoveringTransformations	= 0;
	for (i = 0; i < theMaxNumReflectingCoveringTransformations; i++)
		if (theReflectingCoveringTransformations[i].columns[3][3] == 1.0)
			theNumReflectingCoveringTransformations++;
	
	
	//	The matrix its3DTilingIntoFrameCell has determinant
	//
	//		-1 if the tiling's placement in the frame cell is reflected, or
	//		+1 if it's not reflected.
	//
	theTilingIntoFrameCellDeterminant = Matrix44EuclideanDeterminant(md->its3DTilingIntoFrameCell);

	//	The front-face winding direction ultimately depends on the composition of
	//
	//		- the game-cell-into-tiling mapping (the covering transformation) with
	//		- the tiling-into-frame-cell mapping.
	//
	//	Here we are assuming that
	//
	//		- the object-into-game-cell mapping never reflects, and
	//		- the frame-cell-into-world mapping never reflects;
	//
	//	other we'd need to factor them in too.
	//
	if (theTilingIntoFrameCellDeterminant > 0.0)
	{
		theWindingForPlainCoveringTransformations		= MTLWindingClockwise;	//	This is Metal's default value.
		theWindingForReflectingCoveringTransformations	= MTLWindingCounterClockwise;
	}
	else
	{
		theWindingForPlainCoveringTransformations		= MTLWindingCounterClockwise;
		theWindingForReflectingCoveringTransformations	= MTLWindingClockwise;	//	When the game-cell-placement-in-tiling and
																				//	the tiling-placement-in-world both reflect,
																				//	then their composition, namely the game-cell-placement-in-world
																				//	comes out plain (non-reflecting).
	}

	//	Set common state
	[aRenderEncoder setVertexBuffer:the3DUniformBuffer offset:0 atIndex:BufferIndexVFWorldData];

	//	Draw the walls
	//
	//		Note:  The method -encodeCommandsFor3DWallsWithEncoder:… will
	//		set the renderPipelineState on a per wall basis,
	//		to facilitate the algorithmic fragment shaders
	//		that draw patterns on the walls to visually show the topology.
	//
	[aRenderEncoder setDepthStencilState:its3DDepthStencilStateNormal];
	[aRenderEncoder setCullMode:MTLCullModeNone];	//	will need to show walls' backfaces during reset-game animations
	[self encodeCommandsFor3DWallsWithEncoder:aRenderEncoder uniformBuffer:the3DUniformBuffer modelData:md];
	
	//	Locate the three parts of the3DPolyhedronPlacementBuffer.

	theSolidBufferLength					= GetNum3DPolyhedra_BufferPart1_Solids(md);
	theRectangularSliceBufferLength			= GetNum3DPolyhedra_BufferPart2_RectangularSlices(md);
	theCircularSliceBufferLength			= GetNum3DPolyhedra_BufferPart3_CircularSlices(md);
	theClippedEllipticalSliceBufferLength	= GetNum3DPolyhedra_BufferPart4_ClippedEllipticalSlices(md);

	theSolidBufferOffset					= 0;
	theRectangularSliceBufferOffset			= theSolidBufferOffset				+ theSolidBufferLength;
	theCircularSliceBufferOffset			= theRectangularSliceBufferOffset	+ theRectangularSliceBufferLength;
	theClippedEllipticalSliceBufferOffset	= theCircularSliceBufferOffset		+ theCircularSliceBufferLength;

	//	Draw the game content

	[aRenderEncoder setVertexBuffer:the3DPolyhedronPlacementBuffer
						offset:0
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setCullMode:MTLCullModeBack];	//	never need to show game contents' backfaces

	//	The game content itself

	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedronClippedToFrameCell];
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("invalid ViewType in 3D game");
			break;

		case ViewRepeating:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedron];	//	no special clipping needed
			break;
	}
	[aRenderEncoder setDepthStencilState:its3DDepthStencilStateNormal];

	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			if (theNumPlainCoveringTransformations      > 0)	//	always true
			{
				[aRenderEncoder setVertexBuffer:the3DPlainCoveringTransformationBuffer
									offset:0
									atIndex:BufferIndexVFCoveringTransformations];
				[aRenderEncoder setFrontFacingWinding:theWindingForPlainCoveringTransformations];
				[self encodeCommandsFor3DTicTacToeWithEncoder:	aRenderEncoder
												uniformBuffer:	the3DUniformBuffer
								   numCoveringTransformations:	theNumPlainCoveringTransformations
													modelData:	md];
			}
			if (theNumReflectingCoveringTransformations > 0)	//	true only for Klein space
			{
				[aRenderEncoder setVertexBuffer:the3DReflectingCoveringTransformationBuffer
									offset:0
									atIndex:BufferIndexVFCoveringTransformations];
				[aRenderEncoder setFrontFacingWinding:theWindingForReflectingCoveringTransformations];
				[self encodeCommandsFor3DTicTacToeWithEncoder:	aRenderEncoder
												uniformBuffer:	the3DUniformBuffer
								   numCoveringTransformations:	theNumReflectingCoveringTransformations
													modelData:	md];
			}
			break;

		case Game3DMaze:
			if (theNumPlainCoveringTransformations      > 0)	//	always true
			{
				[aRenderEncoder setVertexBuffer:the3DPlainCoveringTransformationBuffer
									offset:0
									atIndex:BufferIndexVFCoveringTransformations];
				[aRenderEncoder setFrontFacingWinding:theWindingForPlainCoveringTransformations];
				[self      encodeCommandsFor3DMazeWithEncoder:	aRenderEncoder
												uniformBuffer:	the3DUniformBuffer
								   numCoveringTransformations:	theNumPlainCoveringTransformations
													modelData:	md];
			}
			if (theNumReflectingCoveringTransformations > 0)	//	true only for Klein space
			{
				[aRenderEncoder setVertexBuffer:the3DReflectingCoveringTransformationBuffer
									offset:0
									atIndex:BufferIndexVFCoveringTransformations];
				[aRenderEncoder setFrontFacingWinding:theWindingForReflectingCoveringTransformations];
				[self      encodeCommandsFor3DMazeWithEncoder:	aRenderEncoder
												uniformBuffer:	the3DUniformBuffer
								   numCoveringTransformations:	theNumReflectingCoveringTransformations
													modelData:	md];
			}
			break;

		default:
			break;
	}
	
	//	The cross sectional slices where game content intersects the frame cell walls
	//
	//		We've already set buffers as follows:
	//
	//			0	set to the3DUniformBuffer
	//			1	(ignored)
	//			2	set to the3DPolyhedronPlacementBuffer
	//			3	stage_in
	//
	//		No other buffers need to be set.
	//

	//	We'll need to peek at the placement buffer contents,
	//	to see which placements are in use and which are not,
	//	and also to see which are mirror-reversing and which are plain.
	thePlacementBufferContents = [the3DPolyhedronPlacementBuffer contents];

	//	Pipeline state for rectangular and circular cross sections
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedronSliceClippedToFrameCell];
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("invalid ViewType in 3D game");
			break;

		case ViewRepeating:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedronSlice];	//	no special clipping needed
			break;
	}
	[aRenderEncoder setDepthStencilState:its3DDepthStencilStateAlwaysPasses];
	
	//	Rectangular cross sections
	[aRenderEncoder setVertexBuffer:its3DSquareSliceVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	for (i = theRectangularSliceBufferOffset; i < theRectangularSliceBufferOffset + theRectangularSliceBufferLength; i++)
	{
		//		If a placement matrix M   is   in use, then M[3][3] is guaranteed to be 1.0
		//		If a placement matrix M is not in use, then M[3][3] is guaranteed to be 0.0
		if (thePlacementBufferContents[i].itsIsometricPlacement.columns[3][3] == 1.0)
		{
			[aRenderEncoder setFrontFacingWinding:
				EuclideanDeterminantSIMD(&thePlacementBufferContents[i].itsIsometricPlacement) > 0.0 ?
					MTLWindingClockwise :			//	Metal's default value
					MTLWindingCounterClockwise];	//	opposite of default value
			
			[aRenderEncoder	setVertexBufferOffset:(i * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
								atIndex:BufferIndexVFPlacement];
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							vertexStart:	0
							vertexCount:	NUM_SQUARE_SLICE_VERTICES];
		}
	}

	//	Set an appropriate-resolution mesh for circular and elliptical cross sections.
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			theNumCircularSliceVertices		= gCircularSliceNumVertices[MESH_REFINEMENT_LEVEL_FOR_HIGH_RESOLUTION];
			theCircularSliceVertexBuffer	= its3DCircularSliceVertexBufferHighRes;
			break;
			
		case ViewRepeating:
			theNumCircularSliceVertices		= gCircularSliceNumVertices[MESH_REFINEMENT_LEVEL_FOR_LOW_RESOLUTION ];
			theCircularSliceVertexBuffer	= its3DCircularSliceVertexBufferLowRes;
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("ViewBasicSmall not used with 3D games");
			break;

		default:
			GEOMETRY_GAMES_ABORT("invalid view type");
			break;
	}
	
	//	Circular cross sections (with no "extra" clipping)
	[aRenderEncoder setVertexBuffer:theCircularSliceVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	for (i = theCircularSliceBufferOffset; i < theCircularSliceBufferOffset + theCircularSliceBufferLength; i++)
	{
		//		If a placement matrix M   is   in use, then M[3][3] is guaranteed to be 1.0
		//		If a placement matrix M is not in use, then M[3][3] is guaranteed to be 0.0
		if (thePlacementBufferContents[i].itsIsometricPlacement.columns[3][3] == 1.0)
		{
			[aRenderEncoder setFrontFacingWinding:
				EuclideanDeterminantSIMD(&thePlacementBufferContents[i].itsIsometricPlacement) > 0.0 ?
					MTLWindingClockwise :			//	Metal's default value
					MTLWindingCounterClockwise];	//	opposite of default value

			[aRenderEncoder	setVertexBufferOffset:(i * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
								atIndex:BufferIndexVFPlacement];
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							vertexStart:	0
							vertexCount:	theNumCircularSliceVertices];
		}
	}

	//	Pipeline state for elliptical cross sections
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedronSliceClippedToFrameCellAndWinLine];
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("invalid ViewType in 3D game");
			break;

		case ViewRepeating:
			[aRenderEncoder setRenderPipelineState:itsRenderPipelineState3DPolyhedronSliceClippedToWinLine];
			break;
	}
	
	//	Elliptical cross sections (clipped to respect ends of win line)
	[aRenderEncoder setVertexBuffer:theCircularSliceVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	for (i = theClippedEllipticalSliceBufferOffset; i < theClippedEllipticalSliceBufferOffset + theClippedEllipticalSliceBufferLength; i++)
	{
		//		If a placement matrix M   is   in use, then M[3][3] is guaranteed to be 1.0
		//		If a placement matrix M is not in use, then M[3][3] is guaranteed to be 0.0
		if (thePlacementBufferContents[i].itsIsometricPlacement.columns[3][3] == 1.0)
		{
			[aRenderEncoder setFrontFacingWinding:
				EuclideanDeterminantSIMD(&thePlacementBufferContents[i].itsIsometricPlacement) > 0.0 ?
					MTLWindingClockwise :			//	Metal's default value
					MTLWindingCounterClockwise];	//	opposite of default value

			[aRenderEncoder	setVertexBufferOffset:(i * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
								atIndex:BufferIndexVFPlacement];
			[aRenderEncoder	drawPrimitives:	MTLPrimitiveTypeTriangleStrip
							vertexStart:	0
							vertexCount:	theNumCircularSliceVertices];
		}
	}
}

- (void) encodeCommandsFor3DWallsWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							   uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
								   modelData:	(ModelData *)md
{
	id<MTLBuffer>				theVertexBuffer;
	NSUInteger					theNumVertices;
	double						theFrameCellIntoWorld[4][4];
	unsigned int				i;
	const double				*theNormalVector;
	id<MTLRenderPipelineState>	theCurrentRenderPipelineState,
								theDesiredRenderPipelineState;
	bool						theResetAsDomainIsInProgress,
								theDrawWallFrontside,
								theDrawWallBackside,
								thePatternParity;

	if (md->its3DCurrentAperture == 1.0)	//	If aperture is fully open...
		return;								//	...there's nothing to draw.

	if (md->its3DCurrentAperture == 0.0)
		theVertexBuffer = its3DWallVertexBuffer;
	else
		theVertexBuffer = its3DWallWithApertureVertexBuffer;

	theNumVertices = [theVertexBuffer length] / sizeof(TorusGames3DWallVertexData);

	//	The vertex function will get itsFrameCellIntoWorld via the TorusGames3DUniformData,
	//	but we also need a copy of it here, to pass into Wall3DShouldBeTransparent().
	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);
	
	theCurrentRenderPipelineState = nil;

	[aRenderEncoder setVertexBuffer:its3DWallPlacements	offset:0 atIndex:BufferIndexVFPlacement];
	[aRenderEncoder setVertexBuffer:theVertexBuffer		offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFMisc];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:0	//	offset to be set below
						atIndex:BufferIndexVFTexCoordShift];

	for (i = 0; i < NUM_FRAME_WALLS; i++)
	{
		//	The wall placement's third row is exactly
		//	the wall's outward-pointing normal vector,
		//	which, along with the game topology, determines whether
		//	to paint a plain, reflected, quarter-turn or half-turn
		//	pattern on that wall.
		theNormalVector = gWallIntoFrameCellPlacements[i][2];

		//	Should we draw this wall?
		
		//		Is the spinning frame cell animation running?
		theResetAsDomainIsInProgress = (
				md->itsSimulationStatus == Simulation3DResetGameAsDomainPart1
			 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart2
			 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart3);
		
		//		Does the user see the wall's inner face??
		theDrawWallFrontside = ! Wall3DShouldBeTransparent(	theNormalVector,
															theFrameCellIntoWorld,
															theResetAsDomainIsInProgress ?
																md->its3DResetGameScaleFactor : 1.0);

		//		If the user doesn't see the inner face, should we show the outer face?
		theDrawWallBackside = (
				! theDrawWallFrontside				//	inner face is not visible
			 && theResetAsDomainIsInProgress		//	reset is in progress
			 && md->its3DResetGameOpaqueWalls[i]);	//	wall was initially opaque
		
		//		Skip walls that should be transparent.
		if ( ! (theDrawWallFrontside || theDrawWallBackside) )
			continue;


		//	Select a pipeline state based on itsTopology and which wall we're drawing.
		//
		//	Most players will play in TopologyTorus, for which the following code
		//	will set itsRenderPipelineState3DWallPlain once and only once.
		//	For the other topologies, two different pipeline states will get used.
		//
		switch (md->itsTopology)
		{
			case Topology3DTorus:
				theDesiredRenderPipelineState = itsRenderPipelineState3DWallPlain;
				break;
			
			case Topology3DKlein:
				theDesiredRenderPipelineState = (
					//	does the wall's normal vector point in the ±y direction?
					theNormalVector[1] != 0.0 ?
						itsRenderPipelineState3DWallReflection :
						itsRenderPipelineState3DWallPlain);
				break;
			
			case Topology3DQuarterTurn:
				theDesiredRenderPipelineState = (
					//	does the wall's normal vector point in the ±z direction?
					theNormalVector[2] != 0.0 ?
						itsRenderPipelineState3DWallQuarterTurn :
						itsRenderPipelineState3DWallPlain);
				break;
			
			case Topology3DHalfTurn:
				theDesiredRenderPipelineState = (
					//	does the wall's normal vector point in the ±z direction?
					theNormalVector[2] != 0.0 ?
						itsRenderPipelineState3DWallHalfTurn :
						itsRenderPipelineState3DWallPlain);
				break;
			
			default:
				GEOMETRY_GAMES_ABORT("Invalid topology in encodeCommandsFor3DWallsWithEncoder");
				break;
		}
		GEOMETRY_GAMES_ASSERT(theDesiredRenderPipelineState != nil, "-");	//	keep Xcode's static analyzer happy
		if (theCurrentRenderPipelineState != theDesiredRenderPipelineState)
		{
			[aRenderEncoder setRenderPipelineState:theDesiredRenderPipelineState];
			theCurrentRenderPipelineState = theDesiredRenderPipelineState;
		}
		
		//	Quarter-turn and half-turn wall pairs need to invert the color pattern
		//	on one face relative to the other.
		if (theCurrentRenderPipelineState == itsRenderPipelineState3DWallQuarterTurn
		 || theCurrentRenderPipelineState == itsRenderPipelineState3DWallHalfTurn)
		{
			//	The easiest way to set the parity is to check
			//	the z-coordinate of the wall's normal vector,
			//	which will be +1 for one wall but -1 for its mate.
			thePatternParity = (theNormalVector[2] > 0.0);
			
			[aRenderEncoder setFragmentBuffer:	a3DUniformBuffer
									   offset:	thePatternParity ?
													offsetof(TorusGames3DUniformData, itsPatternParityPlus) :
													offsetof(TorusGames3DUniformData, itsPatternParityMinus)
									  atIndex:	BufferIndexFFMisc];
		}
		
		[aRenderEncoder setVertexBufferOffset:(i * sizeof(simd_float4x4))
							atIndex:BufferIndexVFPlacement];
		[aRenderEncoder setVertexBufferOffset:offsetof(TorusGames3DUniformData, itsWallColors[i/2])	//	integer division discards remainder
							atIndex:BufferIndexVFMisc];
		[aRenderEncoder setVertexBufferOffset:offsetof(TorusGames3DUniformData, itsTexCoordShifts[i])
							atIndex:BufferIndexVFTexCoordShift];

		[aRenderEncoder drawPrimitives:	MTLPrimitiveTypeTriangleStrip
						   vertexStart:	0
						   vertexCount:	theNumVertices
						 instanceCount:	1];
	}
}


#pragma mark -
#pragma mark 3D game-specific rendering

- (void)encodeCommandsFor3DTicTacToeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
								  uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
					 numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
									  modelData:	(ModelData *)md
{
	MeshBufferPair	*theBallMesh,
					*theTubeMesh;
	unsigned int	theCount,
					i,
					j,
					k;
	TicTacToePlayer	theActualOwner,
					thePotentialOwner;
	MeshBufferPair	*theMesh;
	NSUInteger		theColorOffset;
	
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			theBallMesh	= its3DBallMeshHighRes;
			theTubeMesh	= its3DTubeMeshHighRes;
			break;
			
		case ViewRepeating:
			theBallMesh	= its3DBallMeshLowRes;	//	As of May 2017, only low res runs at an acceptable frame rate.
			theTubeMesh	= its3DTubeMeshLowRes;	//	Keep the tube mesh consistent with the ball mesh, so they fit snugly.
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("ViewBasicSmall not used with 3D games");
			break;

		default:
			GEOMETRY_GAMES_ABORT("invalid view type");
			break;
	}

	//	The buffer at index BufferIndexVFMisc provides the desired color.
	//	For now just set the buffer, without worrying about the offset.
	//	As the desired color changes, the loop below will set the appropriate offset.
	//
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
		offset:0	//	offset to be set below
		atIndex:BufferIndexVFMisc];

	theCount = 0;

	//	markers
	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				theActualOwner		= md->itsGameOf.TicTacToe3D.itsBoard[i][j][k];
				thePotentialOwner	= (theActualOwner != PlayerNone ?
										theActualOwner : md->itsGameOf.TicTacToe3D.itsWhoseTurn);
				
				if (
#ifdef MAKE_GAME_CHOICE_ICONS
					//	When making a game choice icon, omit the markers
					//	at unoccupied nodes whether or not the game is over,
					//	to keep the icon simple.
#else
					//	When the game is over, don't draw markers at unoccupied nodes.
					md->itsGameIsOver
				 &&
#endif
					theActualOwner == PlayerNone)
				{
				}
				else
				{
					[aRenderEncoder	setVertexBufferOffset:(theCount * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
						atIndex:BufferIndexVFPlacement];
					
					switch (thePotentialOwner)
					{
						case PlayerNone:	GEOMETRY_GAMES_ABORT("invalid potential owner"); theMesh = nil;	break;
						case PlayerX:		theMesh = its3DCubeMesh;	break;
						case PlayerO:		theMesh = theBallMesh;		break;
					}
					[aRenderEncoder setVertexBuffer:theMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];

					switch (theActualOwner)
					{
						case PlayerNone:	theColorOffset = offsetof(TorusGames3DUniformData, itsTicTacToeNeutralColor	);	break;
						case PlayerX:		theColorOffset = offsetof(TorusGames3DUniformData, itsTicTacToeXColor		);	break;
						case PlayerO:		theColorOffset = offsetof(TorusGames3DUniformData, itsTicTacToeOColor		);	break;
					}
					[aRenderEncoder setVertexBufferOffset:theColorOffset
										atIndex:BufferIndexVFMisc];

					[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
									indexCount:				3 * theMesh->itsNumFaces
									indexType:				MTLIndexTypeUInt16
									indexBuffer:			theMesh->itsIndexBuffer
									indexBufferOffset:		0
									instanceCount:			aNumCoveringTransformations];
				}
				
				theCount++;
			}
		}
	}

	//	win line (if needed)
	if (md->itsGameIsOver)
	{
		[aRenderEncoder setVertexBufferOffset:offsetof(TorusGames3DUniformData, itsTicTacToeWinLineColor)
							atIndex:BufferIndexVFMisc];

		//	the tube
		[aRenderEncoder setVertexBuffer:theTubeMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
		[aRenderEncoder	setVertexBufferOffset:
			((TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE) * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
			atIndex:BufferIndexVFPlacement];
		[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
						indexCount:				3 * theTubeMesh->itsNumFaces
						indexType:				MTLIndexTypeUInt16
						indexBuffer:			theTubeMesh->itsIndexBuffer
						indexBufferOffset:		0
						instanceCount:			aNumCoveringTransformations];

		//	the endcaps (if needed)
		if ( ! WinLineIsCircular(md->itsTopology, &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow) )
		{
			[aRenderEncoder setVertexBuffer:theBallMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
			for (i = 0; i < 2; i++)
			{
				[aRenderEncoder	setVertexBufferOffset:
					((TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE + 1 + i) * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
					atIndex:BufferIndexVFPlacement];
				[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
								indexCount:				3 * theBallMesh->itsNumFaces
								indexType:				MTLIndexTypeUInt16
								indexBuffer:			theBallMesh->itsIndexBuffer
								indexBufferOffset:		0
								instanceCount:			aNumCoveringTransformations];
			}
		}
	}
}

- (void)encodeCommandsFor3DMazeWithEncoder:	(id<MTLRenderCommandEncoder>)aRenderEncoder
							 uniformBuffer:	(id<MTLBuffer>)a3DUniformBuffer
				numCoveringTransformations:	(NSUInteger)aNumCoveringTransformations
								 modelData:	(ModelData *)md
{
	MeshBufferPair	*theBallMesh,
					*theTubeMesh;
	unsigned int	n,
					theCount,
					i,
					j,
					k;
	Maze3DNodeEdges	*theNodeEdges;
	unsigned int	theNumTubes,
					theSphereFraction;	//	= 2, 4 or 8, meaning 1/2, 1/4 or 1/8 of a sphere, respectively


	n = md->itsGameOf.Maze3D.itsSize;

	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			theBallMesh	= its3DBallMeshHighRes;
			theTubeMesh	= its3DTubeMeshHighRes;
			break;
			
		case ViewRepeating:
			theBallMesh	= its3DBallMeshLowRes;	//	As of May 2017, only low res runs at an acceptable frame rate.
			theTubeMesh	= its3DTubeMeshLowRes;	//	Keep the tube mesh consistent with the ball mesh, so they fit snugly.
			break;

		case ViewBasicSmall:
			GEOMETRY_GAMES_ABORT("ViewBasicSmall not used with 3D games");
			break;

		default:
			GEOMETRY_GAMES_ABORT("invalid view type");
			break;
	}

	theCount = 0;	//	A single count runs over all the nodes, tubes, slider and goal, in that order.

	//	nodes
	[aRenderEncoder setVertexBuffer:theBallMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:offsetof(TorusGames3DUniformData, itsMazeTracksColor)
						atIndex:BufferIndexVFMisc];
	for (i = 0; i < n; i++)
	{
		for (j = 0; j < n; j++)
		{
			for (k = 0; k < n; k++)
			{
				theNodeEdges = &md->itsGameOf.Maze3D.itsEdges[i][j][k];

				//	If a tube goes straight through the node in some direction...
				if ((theNodeEdges->itsOutbound[0] && theNodeEdges->itsInbound[0])
				 || (theNodeEdges->itsOutbound[1] && theNodeEdges->itsInbound[1])
				 || (theNodeEdges->itsOutbound[2] && theNodeEdges->itsInbound[2]))
				{
					//	...then no sphere need be drawn at this node.
				}
				else	//	Otherwise no tube goes straight through...
				{
					//	...and we'll need to draw a 1/2 sphere, 1/4 sphere or 1/8 sphere
					//	according to whether 1, 2 or 3 tubes meet at this node.
					theNumTubes	= (theNodeEdges->itsOutbound[0] ? 1 : 0)
								+ (theNodeEdges->itsInbound [0] ? 1 : 0)
								+ (theNodeEdges->itsOutbound[1] ? 1 : 0)
								+ (theNodeEdges->itsInbound [1] ? 1 : 0)
								+ (theNodeEdges->itsOutbound[2] ? 1 : 0)
								+ (theNodeEdges->itsInbound [2] ? 1 : 0);
					switch (theNumTubes)
					{
						case 1:	theSphereFraction = 2;	break;	//	1/2 sphere
						case 2:	theSphereFraction = 4;	break;	//	1/4 sphere
						case 3:	theSphereFraction = 8;	break;	//	1/8 sphere
						default:
							GEOMETRY_GAMES_ABORT("Impossible number of tubes meet at node.");
							break;
					}

					//	The platform-independent function Get3DMazePolyhedronPlacementsForSolids()
					//	has already premultiplied the polyhedron placement by an appropriate rotation
					//	so that the visible 1/2, 1/4 or 1/8 sphere will appear in the correct place.
					[aRenderEncoder	setVertexBufferOffset:(theCount * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
						atIndex:BufferIndexVFPlacement];

					[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
									indexCount:				(3 * theBallMesh->itsNumFaces) / theSphereFraction
									indexType:				MTLIndexTypeUInt16
									indexBuffer:			theBallMesh->itsIndexBuffer
									indexBufferOffset:		0
									instanceCount:			aNumCoveringTransformations];
				}

				theCount++;
			}
		}
	}
	
	//	tubes
	[aRenderEncoder setVertexBuffer:theTubeMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:offsetof(TorusGames3DUniformData, itsMazeTracksColor)
						atIndex:BufferIndexVFMisc];
	for (i = n*n*n - 1; i-- > 0; )
	{
		[aRenderEncoder	setVertexBufferOffset:(theCount * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
			atIndex:BufferIndexVFPlacement];

		[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
						indexCount:				(3 * theTubeMesh->itsNumFaces)
						indexType:				MTLIndexTypeUInt16
						indexBuffer:			theTubeMesh->itsIndexBuffer
						indexBufferOffset:		0
						instanceCount:			aNumCoveringTransformations];

		theCount++;
	}
	
	//	slider
	[aRenderEncoder setVertexBuffer:theBallMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:offsetof(TorusGames3DUniformData, itsMazeSliderColor)
						atIndex:BufferIndexVFMisc];
	[aRenderEncoder	setVertexBufferOffset:(theCount * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
						atIndex:BufferIndexVFPlacement];
	[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
					indexCount:				(3 * theBallMesh->itsNumFaces)
					indexType:				MTLIndexTypeUInt16
					indexBuffer:			theBallMesh->itsIndexBuffer
					indexBufferOffset:		0
					instanceCount:			aNumCoveringTransformations];
	theCount++;
	
	//	goal

	[aRenderEncoder	setVertexBufferOffset:(theCount * sizeof(TorusGames3DPolyhedronPlacementAsSIMD))
						atIndex:BufferIndexVFPlacement];

	//		outer surface
	[aRenderEncoder setVertexBuffer:its3DCubeSkeletonOuterFaceMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:offsetof(TorusGames3DUniformData, itsMazeGoalOuterColor)
						atIndex:BufferIndexVFMisc];
	[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
					indexCount:				(3 * its3DCubeSkeletonOuterFaceMesh->itsNumFaces)
					indexType:				MTLIndexTypeUInt16
					indexBuffer:			its3DCubeSkeletonOuterFaceMesh->itsIndexBuffer
					indexBufferOffset:		0
					instanceCount:			aNumCoveringTransformations];

	//		inner surface
	[aRenderEncoder setVertexBuffer:its3DCubeSkeletonInnerFaceMesh->itsVertexBuffer offset:0 atIndex:BufferIndexVFVertexAttributes];
	[aRenderEncoder setVertexBuffer:a3DUniformBuffer
						offset:offsetof(TorusGames3DUniformData, itsMazeGoalInnerColor)
						atIndex:BufferIndexVFMisc];
	[aRenderEncoder	drawIndexedPrimitives:	MTLPrimitiveTypeTriangle
					indexCount:				(3 * its3DCubeSkeletonInnerFaceMesh->itsNumFaces)
					indexType:				MTLIndexTypeUInt16
					indexBuffer:			its3DCubeSkeletonInnerFaceMesh->itsIndexBuffer
					indexBufferOffset:		0
					instanceCount:			aNumCoveringTransformations];

	theCount++;
}


@end


static simd_float3x3 ConvertPlacementToSIMD(
	const Placement2D	*aPlacement)
{
	double			h,
					v,
					c,
					s,
					f,
					a,
					b;
	simd_float3x3	m;

	//	Represent aPlacement as a 3×3 matrix.
	//	The matrix should allow for ordinary vertices
	//
	//		       (h 0 0)( c s 0)(f 0 0)(1 0 0)
	//		(x y 1)(0 v 0)(-s c 0)(0 1 0)(0 1 0)
	//		       (0 0 1)( 0 0 1)(0 0 1)(a b 1)
	//		          ↑       ↑      ↑      ↑
	//			  dilation rotation flip translation
	//			   factor   factor  factor factor
	//
	//	Caution:  A matrice's columns in SIMD's right-to-left convention
	//	appear as the rows in our left-to-right convention.

	h = aPlacement->itsSizeH;
	v = aPlacement->itsSizeV;
	if (aPlacement->itsAngle != 0.0)
	{
		c = cos(aPlacement->itsAngle);
		s = sin(aPlacement->itsAngle);
	}
	else
	{
		c = 1.0;
		s = 0.0;
	}
	f = (aPlacement->itsFlip ? -1.0 : 1.0);
	a = aPlacement->itsH;
	b = aPlacement->itsV;

	m.columns[0][0] = h * c    * f;
	m.columns[0][1] = h * s;
	m.columns[0][2] = 0.0;
	
	m.columns[1][0] = v * (-s) * f;
	m.columns[1][1] = v * c;
	m.columns[1][2] = 0.0;
	
	m.columns[2][0] = a;
	m.columns[2][1] = b;
	m.columns[2][2] = 1.0;
	
	return m;
}
